/**
 * @author: nordsturm
 * © Yandex LLC.
 */

#ifndef YMOD_LEASE_RESOURCE_SERVER_H
#define YMOD_LEASE_RESOURCE_SERVER_H

#include <boost/enable_shared_from_this.hpp>
#include "types_priv.h"
#include "messages.h"
#include "time_millis.h"
#include "timers_queue.h"
#include "logger.h"

namespace ylease {

typedef boost::function<void(const promise_msg&)> promise_callback;
typedef boost::function<void(const reject_msg&)> reject_callback;
typedef boost::function<void(const accepted_msg&)> accepted_callback;

/** Resource server
 *
 *
 * State machine (per resource), states: free, busy.
 *
 * events:
 *   - prepare
 *   - accept
 *   - lease timeout
 *
 * timers:
 *   - "lease expiried", is being created on successfull accept
 */
class resource_server : public boost::enable_shared_from_this<resource_server>
{
public:
    resource_server(
        const string& arbiter_id,
        const string& resource_name,
        timers::queue_interface& timers_queue,
        const log_function_t& log_func);
    ~resource_server();

    void fini();

    bool is_free() const;
    bool is_busy() const;

    void prepare(
        const prepare_msg& request,
        const promise_callback& promise_cb,
        const reject_callback& reject_cb);

    void accept(
        const accept_msg& request,
        const accepted_callback& accepted_cb,
        const reject_callback& reject_cb);

    void timeout(ballot_t ballot);

private:
    bool is_busy_impl() const;

    void switch_to_busy(ballot_t ballot, const lease& lease);

    void switch_to_free();

    bool accepted_expired(time_millis now);

    time_millis accepted_expires(time_millis now);

    void reset_timer(const time_duration& interval, ballot_t ballot);
    void cancel_timer();

    void timeout_impl(ballot_t ballot);

    void update_highest_ballot(ballot_t ballot);

    mutable mutex mux_;
    string arbiter_id_;
    string resource_name_;
    ballot_t highest_promised;
    ballot_t accepted_ballot;
    lease accepted_lease;
    time_millis accepted_expire_time;

    timers::timer_ptr timer_;
    timers::queue_interface& timers_queue_;
    log_function_t log_func_;
};

}

#endif
