#pragma once

#include <ymod_httpclient/call.h>
#include <sharpei_client/sharpei_client.h>

#include <yplatform/util/sstream.h>
#include <boost/algorithm/string.hpp>

#include <memory>

namespace ymod_macs {

class http_client : public sharpei::client::http::HttpClient
{
    using timeout = sharpei::client::http::Timeout;
    using arguments = sharpei::client::http::Arguments;
    using address = sharpei::client::http::Address;
    using headers = sharpei::client::http::Headers;

    using handler = sharpei::client::http::ResponseHandler;

public:
    http_client(yplatform::task_context_ptr context, const std::string& http_module_name)
        : context_(context)
    {
        http_module_ = yplatform::find<yhttp::call>(http_module_name);
    }

    void aget(
        const address& addr,
        timeout /* timeout */,
        const std::string& method,
        const arguments& args,
        const headers& headers,
        handler handler,
        bool /*keepAlive*/,
        const std::string& /* requestId */) const override
    {
        auto url = make_url(addr, method, args);
        auto str_headers = make_headers(headers);

        http_module_->async_run(
            context_, yhttp::request::GET(url, str_headers), [handler](auto err, auto resp) {
                handler(err, { static_cast<unsigned>(resp.status), resp.body });
            });
    }

    void apost(
        const address& addr,
        timeout /*timeout*/,
        const std::string& method,
        const arguments& args,
        const headers& headers,
        const std::string& data,
        handler handler,
        bool /*keepAlive*/,
        const std::string& /* requestId */
    ) const override
    {
        auto url = make_url(addr, method, args);
        auto str_headers = make_headers(headers);
        http_module_->async_run(
            context_,
            yhttp::request::POST(url, str_headers, std::string(data)),
            [handler](auto err, auto resp) {
                handler(err, { static_cast<unsigned>(resp.status), resp.body });
            });
    }

private:
    std::string make_url(const address& addr, const std::string& method, const arguments& args)
        const
    {
        std::vector<std::string> pairs;
        for (auto arg : args)
        {
            pairs.push_back(arg.first + "=" + boost::join(arg.second, ","));
        }
        std::stringstream urlStream;
        urlStream << addr.host << ":" << addr.port << "/";
        urlStream << std::string(method, 1);
        urlStream << "?" << boost::join(pairs, "&");
        return urlStream.str();
    }

    std::string make_headers(const headers& headers) const
    {
        std::string result;
        yplatform::sstream str(result);
        for (const auto& header : headers)
        {
            str << header.first << ": " << boost::join(header.second, ", ") << "\r\n";
        }
        return result;
    }

private:
    yplatform::task_context_ptr context_;
    boost::shared_ptr<yhttp::call> http_module_;
};

using http_client_ptr = std::shared_ptr<http_client>;

}
