#pragma once

#include "http_client.h"
#include "logger.h"
#include "profiler.h"

#include <ymod_macs/module.h>
#include <macs_pg/macs_pg.h>
#include <pgg/database/statistic.h>
#include <sharpei_client/sharpei_client.h>

#include <yplatform/log.h>
#include <yplatform/reactor.h>
#include <yplatform/find.h>
#include <yplatform/time_traits.h>

namespace ymod_macs {

class module::impl : public yplatform::log::contains_logger
{
public:
    impl(yplatform::reactor& reactor, const yplatform::ptree& conf)
    {
        query_conf_ = macs::pg::readQueryConfFile(conf.get<std::string>("query_conf"));
        credentials_ = { conf.get<std::string>("user"), "" };
        pg_pool_ = macs::pg::ConnectionPoolFactory()
                       .ioService(*(reactor.io()))
                       .connectTimeout(pgg::Milliseconds(conf.get("timeouts.connnect", 1000)))
                       .queryTimeout(pgg::Milliseconds(conf.get("timeouts.query", 500)))
                       .queueTimeout(pgg::Milliseconds(conf.get("timeouts.queue", 500)))
                       .maxConnections(conf.get("max_connections", 30))
                       .asyncResolve(conf.get("async_resolve", false))
                       .ipv6Only(conf.get("ipv6_only", true))
                       .dnsCacheTTL(pgg::Seconds(conf.get("dns.cache_ttl", 120)))
                       .product();

        sharpei_settings_ = read_sharpei_settings(conf.get_child("sharpei"));
        http_module_name_ = conf.get<std::string>("sharpei.http_client_module");
        sharpei_cache_ttl_ = conf.get<yplatform::time_traits::duration>("sharpei.cache_ttl");

        auto ctx = boost::make_shared<yplatform::task_context>();
        http_client_ = std::make_shared<http_client>(ctx, http_module_name_);
        sharpei_client_ = make_sharpei_client(ctx);
    }

    macs_service_ptr get_service(
        yplatform::task_context_ptr context,
        const std::string& uid,
        bool use_replica,
        uid_resolver_factory_ptr factory)
    {
        if (!factory)
        {
            factory = macs::pg::createSharpeiUidResolverFactory(make_sharpei_params(context));
        }
        auto service = (*macs::pg::createSeviceFactory(pg_pool_, factory))
                           .queryConf(query_conf_)
                           .credentials(credentials_)
                           .logger(boost::make_shared<macs_logger>(logger(), uid, context))
                           .profiler(boost::make_shared<macs_profiler>(context))
                           .autoQueryHandleStartegy(
                               use_replica ? macs::pg::readReplicaThenMaster : macs::pg::masterOnly)
                           .product(uid);
        return service;
    }

    macs_shard_ptr get_shard(const shard_id& shard_id, bool use_replica)
    {
        auto factory = macs::pg::createShardResolverFactory(make_sharpei_params());
        auto shard = (*macs::pg::createSeviceFactory(pg_pool_, nullptr, factory))
                         .queryConf(query_conf_)
                         .credentials(credentials_)
                         .logger(boost::make_shared<macs_logger>(logger()))
                         .profiler(boost::make_shared<macs_profiler>())
                         .autoQueryHandleStartegy(
                             use_replica ? macs::pg::readReplicaThenMaster : macs::pg::masterOnly)
                         .shard(shard_id);
        return shard;
    }

    void get_shards(const shards_cb& cb)
    {
        sharpei_client_->asyncStat(cb);
    }

    void get_shard_by_id(const shard_id& id, const shard_cb& cb)
    {
        get_shards([id, cb](mail_errors::error_code err, const id_shard_map& shards) {
            if (err)
            {
                return cb(err, {});
            }
            auto it = shards.find(id);
            if (it != shards.end())
            {
                cb({}, it->second);
            }
            else
            {
                cb(makeErrorCode(sharpei::client::Errors::ShardNotFound), {});
            }
        });
    }

    void get_user_shard_id(
        yplatform::task_context_ptr context,
        const std::string& uid,
        shard_id_cb cb)
    {
        auto sharpei_client = make_sharpei_client(context);
        sharpei_client->asyncGetConnInfo(
            sharpei::client::ResolveParams(uid),
            [cb = std::move(cb)](mail_errors::error_code err, sharpei::client::Shard shard) {
                cb(err, std::move(shard.id));
            });
    }

    yplatform::ptree get_stats() const
    {
        auto pool_stats = pgg::statistic::getPoolStats(*pg_pool_);
        yplatform::ptree pool_stats_node;
        for (auto& [conn_id, pool_stat] : pool_stats)
        {
            yplatform::ptree stats_node;
            stats_node.put("free_connections", pool_stat.freeConnections);
            stats_node.put("busy_connections", pool_stat.busyConnections);
            stats_node.put("pending_connections", pool_stat.pendingConnections);
            stats_node.put("max_connections", pool_stat.maxConnections);
            stats_node.put("queue_size", pool_stat.queueSize);
            stats_node.put(
                "average_request_roundtrip_ms", to_milliseconds(pool_stat.averageRequestRoundtrip));
            stats_node.put(
                "average_request_db_latency_ms",
                to_milliseconds(pool_stat.averageRequestDbLatency));
            stats_node.put("average_wait_time_ms", to_milliseconds(pool_stat.averageWaitTime));
            stats_node.put("dropped_timeout", pool_stat.droppedTimedOut);
            stats_node.put("dropped_failed", pool_stat.droppedFailed);
            stats_node.put("dropped_busy", pool_stat.droppedBusy);
            stats_node.put("dropped_with_result", pool_stat.droppedWithResult);
            pool_stats_node.push_back(std::make_pair(conn_id, stats_node));
        }
        yplatform::ptree stats;
        stats.put_child("pool_stats", pool_stats_node);
        return stats;
    }

private:
    macs::pg::SharpeiParams make_sharpei_params(yplatform::task_context_ptr context = nullptr)
    {
        auto client =
            context ? std::make_shared<http_client>(context, http_module_name_) : http_client_;
        return { client, sharpei_settings_, credentials_ };
    }

    sharpei::client::Settings read_sharpei_settings(const yplatform::ptree& conf)
    {
        sharpei::client::http::Address sharpei_address = { conf.get<std::string>("host"),
                                                           conf.get<unsigned>("port", 80) };

        sharpei::client::Settings sharpei_settings;
        sharpei_settings.sharpeiAddress = sharpei_address;
        sharpei_settings.timeout =
            sharpei::client::http::Timeout(conf.get<unsigned>("timeout", 300));
        sharpei_settings.retries = conf.get<unsigned>("retries", 3u);
        sharpei_settings.keepAlive = conf.get<int>("keepalive", 1);
        return sharpei_settings;
    }

    sharpei::client::SharpeiClientPtr make_sharpei_client(
        yplatform::task_context_ptr context = nullptr)
    {
        auto uniq_id = context ? context->uniq_id() : "";
        auto request_info = sharpei::client::RequestInfo{ uniq_id, "", "", "", uniq_id };
        auto sharpei_client = sharpei::client::createSharpeiClient(
            std::make_shared<http_client>(context, http_module_name_),
            sharpei_settings_,
            request_info);

        return sharpei::client::makeCached(sharpei_client, sharpei_cache_ttl_);
    }

    template <typename Duration>
    long int to_milliseconds(const Duration& d) const
    {
        using namespace yplatform::time_traits;
        return duration_cast<milliseconds>(d).count();
    }

    macs::pg::QueryConf query_conf_;
    macs::pg::Credentials credentials_;
    macs::pg::ConnectionPoolPtr pg_pool_;
    sharpei::client::Settings sharpei_settings_;
    std::string http_module_name_;
    http_client_ptr http_client_;
    yplatform::time_traits::duration sharpei_cache_ttl_;
    sharpei::client::SharpeiClientPtr sharpei_client_;
};

}
