#include <iostream>

#include <ymod_webserver/server.h>
#include <ymod_webserver/response.h>
#include <ymod_webserver/request.h>

#include <yplatform/find.h>
#include <yplatform/module_registration.h>
#include <yplatform/loader.h>

#include <logdog/backend/yplatform_log.h>
#include <logdog/attributes/mail_attributes.h>
#include <logdog/format/tskv.h>

#include <mail/ymod_maildb/include/module.h>

#include <macs_pg/service/factory.h>

#include <user_journal/parameters/mailbox.h>

namespace example {

constexpr static auto formatter = ::logdog::tskv::make_formatter(BOOST_HANA_STRING("mail-example-tskv-log"));

auto getModuleLogger() {
    return ::logdog::make_log(
        formatter,
        std::make_shared<yplatform::log::source>(YGLOBAL_LOG_SERVICE, "example")
    );
}

using ModuleLogger = decltype(getModuleLogger());

using MacsPgLog = macs::pg::logging::v2::TypedLog<ModuleLogger>;


struct Server: public yplatform::module {
    void init(const yplatform::ptree&) {
        auto serverModule = yplatform::find<ymod_webserver::server>("webserver");
        auto maildb = yplatform::find<ymod_maildb::Module>("maildb");
        auto reactor = yplatform::find_reactor<std::shared_ptr>("global");

        serverModule->bind("", {"/reset"}, [=] (ymod_webserver::response_ptr s) {
            boost::asio::spawn(*reactor->io(), [=](boost::asio::yield_context yield) {
                if (auto it = s->request()->url.params.find("uid"); it == s->request()->url.params.end()) {
                    s->result(ymod_webserver::codes::bad_request, "uid is required");
                    return;
                }

                const auto serviceParams = ymod_maildb::parseServiceParams("example", *s);
                const auto journalParams = ymod_maildb::parseUserJournalParams(*s);

                macs::ServicePtr service = maildb->service(
                    serviceParams, journalParams,
                    std::make_shared<MacsPgLog>(getModuleLogger()),
                    macs::pg::masterOnly
                );

                user_journal::Journal journal = maildb->journal(
                    serviceParams, journalParams
                );

                service->envelopes().resetFreshCounter(io_result::make_yield_context(yield));

                namespace ujp = user_journal::parameters;
                journal.write<ujp::ResetFresh>(
                    ujp::id::mdb("pg"),
                    ujp::id::state("custom state"),
                    ujp::id::affected(1ul),
                    ujp::id::hidden(true)
                );

                s->result(ymod_webserver::codes::ok, "ok");
            }, boost::coroutines::attributes(1048576));
        });

        LOGDOG_(getModuleLogger(), notice, logdog::message="example::Server loaded");
    }
};

}

DEFINE_SERVICE_OBJECT(example::Server)


int main(int argc, char* argv[]) {
    if (argc != 2) {
        std::cout << "usage " << argv[0] << " <config>\n";
        return 1;
    }

    return yplatform_start(argv[1]);
}
