#pragma once

#include <butil/network/real_ip_from_headers.h>
#include <pgg/factory.h>


namespace ymod_maildb {

struct ServiceParams {
    std::string uid;
    std::string realIp;
    std::string requestId;
    std::string module;
    std::optional<pgg::Credentials> credentials;

    using UserType = pgg::UidResolver::UserType;
    UserType userType = UserType::existing;
};

struct UserJournalParams {
    std::string connectionId;
    std::string expBoxes;
    std::string enabledExpBoxes;
    std::string clientType;
    std::string clientVersion;
    std::string userAgent;
    std::string yandexUid;
    std::string iCookie;
    std::string sessionInfo;
};

struct MacsServiceParams {
    ymod_maildb::ServiceParams sr;
    ymod_maildb::UserJournalParams uj;
};

template<class Request>
UserJournalParams parseUserJournalParams(const Request& req) {
    const auto header = [&req] (const std::string& name) {
        return req.request()->header_value(name);
    };

    return UserJournalParams {
        .connectionId=header("connection_id"),
        .expBoxes=header("x-yandex-expboxes"),
        .enabledExpBoxes=header("x-yandex-enabledexpboxes"),
        .clientType=header("x-yandex-clienttype"),
        .clientVersion=header("x-yandex-clientversion"),
        .userAgent=header("user-agent"),
        .yandexUid=header("yandexuid"),
        .iCookie=header("icookie"),
        .sessionInfo=req.request()->url.param_value("session_info")
    };
}

template<class Request>
ServiceParams parseServiceParams(const std::string& uid, const std::string& module,
                                 const Request& req) {
    const auto header = [&req] (const std::string& name) -> std::optional<std::string> {
        if (auto it = req.request()->headers.find(name); it != req.request()->headers.end()) {
            return std::make_optional(it->second);
        } else {
            return std::nullopt;
        }
    };

    return ServiceParams {
        .uid=uid,
        .realIp=real_ip::getFromHeaders(
            header("x-forwarded-for"), header("x-real-ip"), req.ctx()->remote_address
        ),
        .requestId=header("x-request-id").value_or(""),
        .module=module
    };
}

template<class Request>
ServiceParams parseServiceParams(const std::string& module, const Request& req) {
    std::string uid = req.request()->url.param_value("uid");
    return parseServiceParams(uid, module, req);
}

}
