#include <mail/ymod_maildb/include/module.h>

#include <macs_pg/integration/pa_profiler.h>

#include <yplatform/module_registration.h>
#include <yplatform/reactor.h>

#include <mail/user_journal/helpers/get_fields.h>

#include <logdog/format/tskv.h>
#include <logdog/attributes/mail_attributes.h>

#include <mail/ymod_maildb/include/internal/user_journal.h>
#include <macs_pg/integration/common_configure.h>
#include <locale>


namespace ymod_maildb {

user_journal::Journal createJournal(const ServiceParams& params,
                                    const UserJournalParams& uj,
                                    user_journal::ServicePtr service,
                                    const user_journal::Uatraits& uatraits,
                                    const user_journal::Geobase& geobase) {
    using namespace user_journal::parameters;
    return service->createJournal(params.uid,
        id::ip(params.realIp),
        id::module(params.module),
        id::sessionInfo(uj.sessionInfo),
        id::suid(""),
        id::yandexuidCookie(uj.yandexUid),
        id::iCookie(uj.iCookie),
        id::connectionId(uj.connectionId),
        id::testBuckets(uj.expBoxes),
        id::enabledTestBuckets(uj.enabledExpBoxes),
        id::clientType(uj.clientType),
        id::clientVersion(uj.clientVersion),
        id::userAgent(uj.userAgent),
        id::requestId(params.requestId),
        user_journal::getFieldsFromUserAgent(uatraits, uj.userAgent),
        user_journal::getFieldsFromGeobase(geobase, params.realIp)
    );
}

user_journal::Journal Module::journal(const ServiceParams& s, const UserJournalParams& uj) const {
    return createJournal(s, uj, userJournalService(), uatraits(), geobase());
}

struct MacsPgWithUserJournal: public Module, public yplatform::module {
    user_journal::ServicePtr userJournalService_;
    macs::pg::integration::CommonConfigure commonConfig_;
    macs::pg::UidResolverFactoryPtr uidResolverFactory_;
    user_journal::Uatraits uatraits_;
    user_journal::Geobase geobase_;
    bool corp_ = false;

    virtual ~MacsPgWithUserJournal() = default;

    void init(const yplatform::ptree& cfg) {
        uatraits_ = user_journal::initUa(cfg.get_child("uatraits"));
        geobase_ = user_journal::initGeodata(cfg);
        corp_ = cfg.get<bool>("corp");

        ModuleLogger moduleLogger = getModuleLogger(cfg.get<std::string>("logger.module"));
        UjLogger ujErrorsLogger = getUjLogger(cfg.get<std::string>("logger.user_journal_errors"));
        auto ujLogger = std::make_shared<yplatform::log::source>(
            YGLOBAL_LOG_SERVICE, cfg.get<std::string>("logger.user_journal")
        );

        const auto writer = [ujLogger](const std::string& line) {
            if (line.empty()) {
                return;
            }

            if (std::size_t p = line.find_last_of('\n'); p != std::string::npos) {
                YLOG(*ujLogger, info) << line.substr(0, p);
            } else {
                YLOG(*ujLogger, info) << line;
            }
        };

        userJournalService_ = user_journal::ServiceFactory()
                    .tskv(cfg.get_child("user_journal"), writer)
                    .profiler(boost::make_shared<ProfilerLog>())
                    .logger(boost::make_shared<UserJournalLog>(ujErrorsLogger))
                    .locale(std::locale(std::locale("ru_RU.UTF-8"), "C", std::locale::numeric))
                    .product();

        yplatform::reactor_ptr reactor;
        bool customIoContext = false;
        if (const auto name = cfg.get_optional<std::string>("dependencies.reactor")) {
            reactor = yplatform::find_reactor(*name);
            customIoContext = true;
        } else {
            reactor = yplatform::global_net_reactor;
        }

        if (reactor->size() != 1 || reactor->get_pool()->size() < 1) {
            throw std::invalid_argument("maildb is optimized for multiple-threads reactor - set "
                                     "pool_count = 1 and io_threads >= 1");
        }

        const auto& metadataCfg = cfg.get_child("metadata");
        if (!metadataCfg.get_child_optional("sharpei.http_client")) {
            throw std::invalid_argument("sharpei.http_client must be set");
        }

        using namespace macs::pg::integration;
        uidResolverFactory_ = CommonConfigure::parseUidResolverFactory(
            metadataCfg, nullptr,
            [moduleLogger] (const std::exception& e) {
                LOGDOG_(moduleLogger, error, log::where_name="sharpei", log::exception=e);
        });
        if (customIoContext) {
            commonConfig_.parse(metadataCfg, reactor->io());
        } else {
            commonConfig_.parse(metadataCfg);
        }

        LOGDOG_(moduleLogger, notice, log::message="ymod_maildb::MacsPgWithUserJournal module initialized");
    }

    macs::ServicePtr service(const ServiceParams& params, const UserJournalParams& uj,
                             macs::pg::logging::v2::LogPtr macsPgLog,
                             const pgg::QueryHandleStrategy& strategy) const override {
        auto journal = createJournal(params, uj, userJournalService_, uatraits_, geobase_);

        auto reqInfo = macs::pg::RequestInfo{params.requestId, uj.connectionId, uj.clientType, params.realIp};
        auto factory = commonConfig_.getFactory(uidResolverFactory_);
        factory->logger(macsPgLog)
                .profiler(macs::pg::integration::PaProfiler::create())
                .userJournal(journal)
                .autoQueryHandleStartegy(strategy)
                .requestInfo(std::move(reqInfo));

        if (params.credentials) {
            factory->credentials(params.credentials.value());
        }
        macs::ServicePtr service = factory->product(params.uid, params.userType);
        service->settings().setMode(corp_ ? macs::settings::Mode::corp
                                          : macs::settings::Mode::production);

        return service;
    }

    user_journal::ServicePtr userJournalService() const override {
        return userJournalService_;
    }

    user_journal::Uatraits uatraits() const override {
        return uatraits_;
    }

    user_journal::Geobase geobase() const override {
        return geobase_;
    }
};

}

DEFINE_SERVICE_OBJECT(ymod_maildb::MacsPgWithUserJournal)
