#include <ymod_mdb_sharder/shards_distributor.h>

#include "shards_distributor_impl.h"
#include "sharpei_client.h"
#include "shards_listener.h"

#include <ymod_lease/ymod_lease.h>
#include <ymod_lease/lock_manager.h>
#include <ymod_lease/bucket_manager.h>

namespace ymod_mdb_sharder {

using lock_manager_type = ylease::lock_manager<ylease::node>;
using bucket_manager_type = ylease::bucket_manager<ylease::node>;
using shards_listener_type = shards_listener<sharpei::client::SharpeiClientPtr>;
using shards_distributor_type = shards_distributor_impl<
    std::shared_ptr<shards_listener_type>,
    std::shared_ptr<lock_manager_type>,
    std::shared_ptr<bucket_manager_type>>;
using shard_bucket_map = std::map<shard_id, bucket_id>;

struct shards_distributor::impl
{
    impl(boost::asio::io_service& io) : io(io)
    {
    }

    boost::asio::io_service& io;
    task_context_ptr ctx = boost::make_shared<yplatform::task_context>();
    std::shared_ptr<shards_listener_type> shards_listener;
    std::shared_ptr<lock_manager_type> lock_manager;
    std::shared_ptr<bucket_manager_type> bucket_manager;
    std::shared_ptr<shards_distributor_type> shards_distributor;
};

shard_bucket_map read_buckets_cfg(const yplatform::ptree& conf)
{
    shard_bucket_map res;
    auto [begin, end] = conf.equal_range("buckets");
    for (auto it = begin; it != end; ++it)
    {
        auto& [tag_name, bucket] = *it;
        auto name = bucket.get<std::string>("name");
        for (auto& [shard_tag_name, shard] :
             boost::make_iterator_range(bucket.equal_range("shards")))
        {
            res[shard.data()] = name;
        }
    }
    return res;
}

shards_distributor::shards_distributor(boost::asio::io_service& io, const yplatform::ptree& conf)
{
    impl_ = std::make_shared<impl>(io);

    auto sharpei_client = create_sharpei_client(impl_->ctx, conf.get_child("sharpei"));
    impl_->shards_listener = std::make_shared<shards_listener_type>(
        &io,
        impl_->ctx,
        conf.get<yplatform::time_traits::duration>(
            "shards_polling_interval", time_traits::seconds(5)),
        sharpei_client);

    std::string my_node_id;
    shard_bucket_map buckets;
    bucket_id open_bucket;
    bool use_lease = conf.get<bool>("use_lease");
    if (use_lease)
    {
        auto lease_name = conf.get<std::string>("lease_module");
        auto lease_node = yplatform::find<ymod_lease::node, std::shared_ptr>(lease_name);
        my_node_id = lease_node->node_id();

        impl_->lock_manager = std::make_shared<lock_manager_type>(
            &io,
            lease_node,
            conf.get<std::size_t>("max_owned_locks"),
            conf.get<std::size_t>("extra_acquire_count", 5u));

        if (conf.count("buckets"))
        {
            buckets = read_buckets_cfg(conf);
            open_bucket = conf.get<std::string>("open_bucket");
            impl_->bucket_manager =
                std::make_shared<bucket_manager_type>(&io, impl_->lock_manager, lease_node);
        }
    }
    else
    {
        std::stringstream stream;
        stream << this;
        my_node_id = stream.str();
    }

    impl_->shards_distributor = std::make_shared<shards_distributor_type>(
        io,
        impl_->ctx,
        buckets,
        open_bucket,
        my_node_id,
        impl_->shards_listener,
        impl_->lock_manager,
        impl_->bucket_manager);
    impl_->shards_distributor->init();
}

void shards_distributor::start()
{
    yplatform::spawn(impl_->io.get_executor(), impl_->shards_listener);
}

void shards_distributor::shards_distributor::logger(const yplatform::log::source& source)
{
    contains_logger::logger(source);
    if (impl_->shards_distributor)
    {
        impl_->shards_distributor->logger(source);
    }
    if (impl_->shards_listener)
    {
        impl_->shards_listener->logger(source);
    }
    if (impl_->lock_manager)
    {
        impl_->lock_manager->logger(source);
    }
    if (impl_->bucket_manager)
    {
        impl_->bucket_manager->logger(source);
    }
}

void shards_distributor::subscribe(
    const shard_ids_cb& on_acquire_shards,
    const shard_ids_cb& on_release_shards)
{
    impl_->shards_distributor->subscribe(on_acquire_shards, on_release_shards);
}

void shards_distributor::get_owner(
    task_context_ptr context,
    const shard_id& shard,
    const node_info_cb& cb)
{
    impl_->shards_distributor->get_owner(context, shard, cb);
}

const std::string& shards_distributor::my_node_id() const
{
    return impl_->shards_distributor->my_node_id();
}

void shards_distributor::get_acquired_shards(const shard_ids_cb& cb)
{
    impl_->shards_distributor->get_acquired_shards(cb);
}

void shards_distributor::get_acquired_buckets_info(const buckets_info_cb& cb)
{
    impl_->shards_distributor->get_acquired_buckets_info(cb);
}

void shards_distributor::release_shards_for(
    const shard_ids& shards,
    const time_traits::duration& duration)
{
    impl_->shards_distributor->release_shards_for(shards, duration);
}

void shards_distributor::release_buckets_for(
    const bucket_ids& buckets,
    const time_traits::duration& duration)
{
    impl_->shards_distributor->release_buckets_for(buckets, duration);
}

void shards_distributor::add_bucket(const bucket_id& bucket, const shard_ids& shards)
{
    impl_->shards_distributor->add_bucket(bucket, shards);
}

void shards_distributor::del_bucket(const bucket_id& bucket)
{
    impl_->shards_distributor->del_bucket(bucket);
}

void shards_distributor::add_shards_to_bucket(const bucket_id& bucket, const shard_ids& shards)
{
    impl_->shards_distributor->add_shards_to_bucket(bucket, shards);
}

void shards_distributor::del_shards_from_bucket(const bucket_id& bucket, const shard_ids& shards)
{
    impl_->shards_distributor->del_shards_from_bucket(bucket, shards);
}

}