#pragma once

#include <ymod_mdb_sharder/types.h>
#include <ymod_mdb_sharder/errors.h>

#include <yplatform/log/contains_logger.h>
#include <yplatform/module.h>
#include <boost/asio/ip/host_name.hpp>

namespace ymod_mdb_sharder {

namespace ph = std::placeholders;

template <typename ShardsListenerPtr, typename LockManagerPtr, typename BucketManagerPtr>
class shards_distributor_impl
    : public std::enable_shared_from_this<
          shards_distributor_impl<ShardsListenerPtr, LockManagerPtr, BucketManagerPtr>>
    , public yplatform::log::contains_logger
{
    using shard_bucket_map = std::map<shard_id, bucket_id>;

public:
    shards_distributor_impl(
        boost::asio::io_service& io,
        task_context_ptr ctx,
        const shard_bucket_map& buckets,
        const bucket_id& open_bucket,
        const std::string& my_node_id,
        ShardsListenerPtr shards_listener,
        LockManagerPtr lock_manager,
        BucketManagerPtr bucket_manager)
        : io_(&io)
        , context_(ctx)
        , shards_listener_(shards_listener)
        , lock_manager_(lock_manager)
        , bucket_manager_(bucket_manager)
        , buckets_cfg_(buckets)
        , open_bucket_(open_bucket)
    {
        my_info_.id = my_node_id;
        my_info_.host = boost::asio::ip::host_name();
    }

    void init()
    {
        if (lock_manager_)
        {
            if (bucket_manager_)
            {
                bucket_manager_->init(
                    io_->wrap(std::bind(
                        &shards_distributor_impl::on_acquire_shard,
                        yplatform::shared_from(this),
                        ph::_1)),
                    io_->wrap(std::bind(
                        &shards_distributor_impl::on_release_shard,
                        yplatform::shared_from(this),
                        ph::_1)));
            }
            else
            {
                lock_manager_->init(
                    io_->wrap(std::bind(
                        &shards_distributor_impl::on_acquire_shard,
                        yplatform::shared_from(this),
                        ph::_1)),
                    io_->wrap(std::bind(
                        &shards_distributor_impl::on_release_shard,
                        yplatform::shared_from(this),
                        ph::_1)));
            }
        }
        shards_listener_->subscribe(
            io_->wrap(std::bind(
                &shards_distributor_impl::on_add_shards, yplatform::shared_from(this), ph::_1)),
            io_->wrap(std::bind(
                &shards_distributor_impl::on_del_shards, yplatform::shared_from(this), ph::_1)));
    }

    void subscribe(const shard_ids_cb& on_acquire_shards, const shard_ids_cb& on_release_shards)
    {
        io_->post(
            [this, self = yplatform::shared_from(this), on_acquire_shards, on_release_shards] {
                on_acquire_shards_.emplace_back(on_acquire_shards);
                on_release_shards_.emplace_back(on_release_shards);
            });
    }

    void get_owner(task_context_ptr context, const shard_id& shard, const node_info_cb& cb)
    {
        if (!lock_manager_ && !bucket_manager_)
        {
            return cb({}, my_info_);
        }
        auto self = yplatform::shared_from(this);
        auto value_cb = [this, self, context, cb](const std::string& value) {
            error_code ec;
            node_info ret;
            try
            {
                if (value.size())
                {
                    ret = ymod_mdb_sharder::node_info::from_string(value);
                }
                else
                {
                    ec = error::not_owned;
                }
            }
            catch (const std::exception& e)
            {
                YLOG_CTX_LOCAL(context, error)
                    << "parse resource value error: " << e.what() << ", value=\"" << value << "\"";
                ec = error::cannot_get_owner;
            }
            cb(ec, std::move(ret));
        };
        if (bucket_manager_)
        {
            bucket_manager_->get_resource_value(shard, value_cb);
        }
        else
        {
            lock_manager_->get_resource_value(shard, value_cb);
        }
    }

    const std::string& my_node_id() const
    {
        return my_info_.id;
    }

    void get_acquired_shards(const shard_ids_cb& cb)
    {
        if (bucket_manager_)
        {
            get_acquired_buckets_info(
                [cb](const std::map<std::string /*bucket*/, shard_ids>& info) {
                    shard_ids shards;
                    for (auto& [bucket, bucket_shards] : info)
                    {
                        shards.insert(shards.end(), bucket_shards.begin(), bucket_shards.end());
                    }
                    cb(shards);
                });
        }
        else if (lock_manager_)
        {
            lock_manager_->get_acquired_resources([cb](const shard_ids& shards) { cb(shards); });
        }
        else
        {
            shards_listener_->get_shards([cb](const shard_ids& shards) { cb(shards); });
        }
    }

    void get_acquired_buckets_info(const buckets_info_cb& cb)
    {
        if (!bucket_manager_)
        {
            return cb({});
        }
        auto self = yplatform::shared_from(this);
        bucket_manager_->get_acquired_buckets(
            io_->wrap([this, self, cb](const bucket_ids& acquired_bucket_ids) {
                std::map<bucket_id, shard_ids> buckets_info;
                for (auto& [shard, bucket] : buckets_cfg_)
                {
                    auto it =
                        std::find(acquired_bucket_ids.begin(), acquired_bucket_ids.end(), bucket);
                    if (it != acquired_bucket_ids.end())
                    {
                        buckets_info[bucket].push_back(shard);
                    }
                }
                cb(buckets_info);
            }));
    }

    void release_shards_for(const shard_ids& shards, const time_traits::duration& duration)
    {
        if (bucket_manager_)
        {
            throw std::runtime_error("release_shards_for not supported while working with buckets");
        }
        if (!lock_manager_)
        {
            throw std::runtime_error("lock_manager not specified");
        }
        lock_manager_->release_resources_for(shards, duration);
    }

    void release_buckets_for(const bucket_ids& buckets, const time_traits::duration& duration)
    {
        if (!bucket_manager_)
        {
            throw std::runtime_error("bucket_manager not specified");
        }
        bucket_manager_->release_buckets_for(buckets, duration);
    }

    void add_bucket(const bucket_id& bucket, const shard_ids& shards)
    {
        if (!bucket_manager_)
        {
            throw std::runtime_error("bucket_manager not specified");
        }
        io_->post([this, self = yplatform::shared_from(this), bucket, shards] {
            for (auto& shard : shards)
            {
                buckets_cfg_[shard] = bucket;
            }
            bucket_manager_->add_buckets({ { bucket, { shards.begin(), shards.end() } } });
        });
    }

    void del_bucket(const bucket_id& bucket)
    {
        if (!bucket_manager_)
        {
            throw std::runtime_error("bucket_manager not specified");
        }
        if (bucket == open_bucket_)
        {
            throw std::runtime_error("cannot delete open_bucket");
        }
        io_->post([this, self = yplatform::shared_from(this), bucket] {
            for (auto it = buckets_cfg_.begin(); it != buckets_cfg_.end();)
            {
                if (it->second == bucket)
                {
                    it = buckets_cfg_.erase(it);
                }
                else
                {
                    ++it;
                }
            }
            bucket_manager_->del_buckets({ bucket });
        });
    }

    void add_shards_to_bucket(const bucket_id& bucket, const shard_ids& shards)
    {
        if (!bucket_manager_)
        {
            throw std::runtime_error("bucket_manager not specified");
        }
        io_->post([this, self = yplatform::shared_from(this), bucket, shards] {
            for (auto& shard : shards)
            {
                buckets_cfg_[shard] = bucket;
            }
            bucket_manager_->add_resources_to_bucket(bucket, { shards.begin(), shards.end() });
        });
    }

    void del_shards_from_bucket(const bucket_id& bucket, const shard_ids& shards)
    {
        if (!bucket_manager_)
        {
            throw std::runtime_error("bucket_manager not specified");
        }
        io_->post([this, self = yplatform::shared_from(this), bucket, shards] {
            for (auto& shard : shards)
            {
                auto it = buckets_cfg_.find(shard);
                if (it != buckets_cfg_.end() && it->second == bucket)
                {
                    buckets_cfg_.erase(it);
                }
            }
            bucket_manager_->del_resources_from_bucket(bucket, { shards.begin(), shards.end() });
        });
    }

private:
    void on_add_shards(const shard_ids& shards)
    {
        if (bucket_manager_)
        {
            for (auto& shard : shards)
            {
                auto it = buckets_cfg_.find(shard);
                auto new_shard = it == buckets_cfg_.end();
                if (new_shard)
                {
                    buckets_cfg_[shard] = open_bucket_;
                }
                auto& bucket = new_shard ? open_bucket_ : it->second;
                bucket_manager_->add_resources_to_bucket(bucket, { shard });
            }
        }
        else if (lock_manager_)
        {
            lock_manager_->on_add_resources(shards);
        }
        else
        {
            for (auto& shard : shards)
            {
                on_acquire_shard(shard);
            }
        }
    }

    void on_del_shards(const shard_ids& shards)
    {
        if (bucket_manager_)
        {
            for (auto& shard : shards)
            {
                auto it = buckets_cfg_.find(shard);
                auto& bucket = it == buckets_cfg_.end() ? open_bucket_ : it->second;
                bucket_manager_->del_resources_from_bucket(bucket, { shard });
            }
        }
        else if (lock_manager_)
        {
            lock_manager_->on_del_resources(shards);
        }
        else
        {
            for (auto& shard : shards)
            {
                on_release_shard(shard);
            }
        }
    }

    void on_acquire_shard(const shard_id& shard)
    {
        try
        {
            if (bucket_manager_)
            {
                bucket_manager_->set_resource_value(shard, my_info_.to_string());
            }
            else if (lock_manager_)
            {
                lock_manager_->set_resource_value(shard, my_info_.to_string());
            }
        }
        catch (const std::exception& e)
        {
            YLOG_CTX_LOCAL(context_, error)
                << "set resource value error: " << e.what() << ", shard=" << shard;
        }
        YLOG_CTX_LOCAL(context_, info) << "shards_distributor_impl::on_acquire_shard " << shard;
        for (auto& cb : on_acquire_shards_)
        {
            cb({ shard });
        }
    }

    void on_release_shard(const shard_id& shard)
    {
        YLOG_CTX_LOCAL(context_, info) << "shards_distributor_impl::on_release_shard " << shard;
        for (auto& cb : on_release_shards_)
        {
            cb({ shard });
        }
    }

    std::vector<shard_ids_cb> on_acquire_shards_;
    std::vector<shard_ids_cb> on_release_shards_;

    boost::asio::io_service* io_ = nullptr;
    task_context_ptr context_;
    ShardsListenerPtr shards_listener_;
    LockManagerPtr lock_manager_;
    BucketManagerPtr bucket_manager_;
    shard_bucket_map buckets_cfg_;
    bucket_id open_bucket_;
    node_info my_info_;
};

}