#pragma once

#include <ymod_mdb_sharder/types.h>

#include <yplatform/module.h>
#include <yplatform/coroutine.h>
#include <yplatform/yield.h>
#include <yplatform/util/sstream.h>
#include <sharpei_client/sharpei_client.h>
#include <set>

namespace ymod_mdb_sharder {

// Should be spawned on single thread io_service
template <typename SharpeiClientPtr>
class shards_listener
    : public std::enable_shared_from_this<shards_listener<SharpeiClientPtr>>
    , public yplatform::log::contains_logger
{
public:
    using yield_context = yplatform::yield_context<shards_listener>;
    using map_shard = sharpei::client::MapShard;

    shards_listener(
        boost::asio::io_service* io,
        task_context_ptr context,
        const yplatform::time_traits::duration& polling_interval,
        SharpeiClientPtr sharpei_client)
        : io_(io)
        , timer_(*io)
        , polling_interval_(polling_interval)
        , sharpei_client_(sharpei_client)
        , context_(context)
    {
    }

    void subscribe(const shard_ids_cb& on_new_shards, const shard_ids_cb& on_remove_shards)
    {
        io_->post([this, self = yplatform::shared_from(this), on_new_shards, on_remove_shards] {
            on_new_shards_.push_back(on_new_shards);
            on_remove_shards_.push_back(on_remove_shards);
        });
    }

    void get_shards(const shard_ids_cb& cb)
    {
        io_->post([this, self = yplatform::shared_from(this), cb] {
            cb({ current_shards_ids_.begin(), current_shards_ids_.end() });
        });
    }

    void operator()(yield_context ctx, mail_errors::error_code err = {}, map_shard shards = {})
    {
        reenter(ctx)
        {
            while (true)
            {
                yield sharpei_client_->asyncStat(ctx);
                try
                {
                    if (err)
                    {
                        YLOG_CTX_LOCAL(context_, error)
                            << "get all shards error: " << err.full_message();
                    }
                    else
                    {
                        YLOG_CTX_LOCAL(context_, debug) << "shards count=" << shards.size();
                        new_shards_ids_.clear();
                        for (auto& [id, shard] : shards)
                        {
                            new_shards_ids_.insert(id);
                        }
                        notify_added_shards();
                        notify_deleted_shards();
                        current_shards_ids_.swap(new_shards_ids_);
                    }
                }
                catch (const std::exception& e)
                {
                    YLOG_CTX_LOCAL(context_, error) << "shards listener exception:" << e.what();
                }
                yield
                {
                    // Ignore exceptions.
                    timer_.expires_from_now(polling_interval_);
                    timer_.async_wait(ctx);
                }
            }
        }
    }

    void operator()(yield_context ctx, const error_code& err)
    {
        if (!err)
        {
            (*this)(ctx);
        }
        else
        {
            YLOG_CTX_LOCAL(context_, warning) << "shard listener aborted: " << err.message();
        }
    }

private:
    void notify_added_shards()
    {
        shard_ids added_shards;
        std::set_difference(
            new_shards_ids_.begin(),
            new_shards_ids_.end(),
            current_shards_ids_.begin(),
            current_shards_ids_.end(),
            std::back_inserter(added_shards));
        if (added_shards.size())
        {
            YLOG_CTX_LOCAL(context_, info) << "add shards " << dump_shards(added_shards);
            for (auto& handler : on_new_shards_)
            {
                handler(added_shards);
            }
        }
    }

    void notify_deleted_shards()
    {
        shard_ids deleted_shards;
        std::set_difference(
            current_shards_ids_.begin(),
            current_shards_ids_.end(),
            new_shards_ids_.begin(),
            new_shards_ids_.end(),
            std::back_inserter(deleted_shards));
        if (deleted_shards.size())
        {
            YLOG_CTX_LOCAL(context_, info) << "del shards " << dump_shards(deleted_shards);
            for (auto& handler : on_remove_shards_)
            {
                handler(deleted_shards);
            }
        }
    }

    std::string dump_shards(const shard_ids& shards)
    {
        std::string dump;
        yplatform::sstream str(dump);
        for (auto& shard : shards)
        {
            str << shard << " ";
        }
        return dump;
    }

    boost::asio::io_service* io_;
    yplatform::time_traits::timer timer_;
    yplatform::time_traits::duration polling_interval_;
    SharpeiClientPtr sharpei_client_;
    std::vector<shard_ids_cb> on_new_shards_;
    std::vector<shard_ids_cb> on_remove_shards_;
    task_context_ptr context_;
    std::set<shard_id> new_shards_ids_;
    std::set<shard_id> current_shards_ids_;
};

}

#include <yplatform/unyield.h>