#pragma once

#include <ymod_mdb_sharder/types.h>

#include <ymod_httpclient/call.h>
#include <yplatform/util/sstream.h>
#include <sharpei_client/sharpei_client.h>
#include <boost/algorithm/string.hpp>

namespace ymod_mdb_sharder {

namespace detail {

class http_client : public sharpei::client::http::HttpClient
{
    using timeout = sharpei::client::http::Timeout;
    using arguments = sharpei::client::http::Arguments;
    using address = sharpei::client::http::Address;
    using headers = sharpei::client::http::Headers;
    using handler = sharpei::client::http::ResponseHandler;

public:
    http_client(task_context_ptr ctx, const std::string& http_module_name) : context_(ctx)
    {
        http_module_ = yplatform::find<yhttp::call>(http_module_name);
    }

    void aget(
        const address& addr,
        timeout /*timeout*/,
        const std::string& method,
        const arguments& args,
        const headers& headers,
        handler handler,
        bool /*keepAlive*/,
        const std::string& /*requestId*/) const override
    {
        auto url = make_url(addr, method, args);
        auto str_headers = make_headers(headers);
        http_module_->async_run(
            context_, yhttp::request::GET(url, str_headers), [handler](auto err, auto resp) {
                handler(err, { static_cast<unsigned>(resp.status), resp.body });
            });
    }

    void apost(
        const address& addr,
        timeout /*timeout*/,
        const std::string& method,
        const arguments& args,
        const headers& headers,
        const std::string& data,
        handler handler,
        bool /*keepAlive*/,
        const std::string& /*requestId*/) const override
    {
        auto url = make_url(addr, method, args);
        auto str_headers = make_headers(headers);
        http_module_->async_run(
            context_,
            yhttp::request::POST(url, str_headers, std::string(data)),
            [handler](auto err, auto resp) {
                handler(err, { static_cast<unsigned>(resp.status), resp.body });
            });
    }

private:
    std::string make_url(const address& addr, const std::string& method, const arguments& args)
        const
    {
        std::vector<std::string> pairs;
        for (auto arg : args)
        {
            pairs.push_back(arg.first + "=" + boost::join(arg.second, ","));
        }
        std::stringstream stream;
        stream << addr.host << ":" << addr.port << "/";
        stream << std::string(method, 1);
        stream << "?" << boost::join(pairs, "&");
        return stream.str();
    }

    std::string make_headers(const headers& headers) const
    {
        std::string result;
        yplatform::sstream str(result);
        for (const auto& header : headers)
        {
            str << header.first << ": " << boost::join(header.second, ", ") << "\r\n";
        }
        return result;
    }

private:
    task_context_ptr context_;
    boost::shared_ptr<yhttp::call> http_module_;
};

}

inline sharpei::client::SharpeiClientPtr create_sharpei_client(
    task_context_ptr ctx,
    const yplatform::ptree& conf)
{
    sharpei::client::Settings settings;
    settings.sharpeiAddress = { conf.get<std::string>("host"), conf.get<unsigned>("port", 80) };
    settings.timeout = sharpei::client::http::Timeout(conf.get<unsigned>("timeout", 300));
    settings.retries = conf.get<unsigned>("retries", 3u);
    auto ttl = conf.get<time_traits::duration>("cache_ttl", time_traits::seconds(5));
    auto http =
        std::make_shared<detail::http_client>(ctx, conf.get<std::string>("http_client_module"));
    auto client = sharpei::client::createSharpeiClient(http, settings, {});
    auto cached_client = sharpei::client::makeCached(client, ttl);
    return cached_client;
}

}