#pragma once

#include "users_listener.h"
#include <ymod_mdb_sharder/errors.h>
#include <ymod_mdb_sharder/types.h>

#include <yplatform/module.h>
#include <sharpei_client/sharpei_client.h>

namespace ymod_mdb_sharder {

template <typename ShardsDistributorPtr, typename SharpeiClientPtr>
class users_distributor_impl
    : public std::enable_shared_from_this<
          users_distributor_impl<ShardsDistributorPtr, SharpeiClientPtr>>
    , public yplatform::log::contains_logger
{
public:
    users_distributor_impl(
        boost::asio::io_service& io,
        task_context_ptr ctx,
        ShardsDistributorPtr shards_distributor,
        SharpeiClientPtr sharpei_client,
        users_listener::settings users_polling_settings)
        : io_(io)
        , context_(ctx)
        , shards_distributor_(shards_distributor)
        , sharpei_client_(sharpei_client)
        , users_polling_settings_(users_polling_settings)
    {
    }

    void init()
    {
        shards_distributor_->subscribe(
            io_.wrap(weak_bind(
                &users_distributor_impl::on_acquire_shards, yplatform::shared_from(this), ph::_1)),
            io_.wrap(weak_bind(
                &users_distributor_impl::on_release_shards, yplatform::shared_from(this), ph::_1)));
    }

    void start()
    {
        if (!get_all_users_ || !get_changed_users_)
        {
            throw std::runtime_error("polling methods not specified");
        }
    }

    void fini()
    {
        for (auto& [uid, listener] : users_listeners_)
        {
            listener->stop();
        }
        users_listeners_.clear();
    }

    void subscribe(
        const shard_id_with_uids_cb& on_acquire_users,
        const shard_id_with_uids_cb& on_release_users)
    {
        io_.post([this, self = yplatform::shared_from(this), on_acquire_users, on_release_users] {
            on_acquire_users_.emplace_back(on_acquire_users);
            on_release_users_.emplace_back(on_release_users);
        });
    }

    void get_owner(task_context_ptr context, uid_t uid, const node_info_cb& cb)
    {
        auto self = yplatform::shared_from(this);
        sharpei_client_->asyncGetConnInfo(
            sharpei::client::ResolveParams(std::to_string(uid)),
            [this, self, context, cb](mail_errors::error_code err, sharpei::client::Shard shard) {
                if (err)
                {
                    YLOG_CTX_LOCAL(context, error) << "get shard error: " << err.full_message();
                    if (err == sharpei::client::Errors::UidNotFound)
                    {
                        return cb(error::user_not_found, node_info());
                    }
                    return cb(err.base(), node_info());
                }
                else
                {
                    shards_distributor_->get_owner(context, shard.id, cb);
                }
            });
    }

    void set_polling_methods(
        const get_all_users_method& get_all_users,
        const get_changed_users_method& get_changed_users)
    {
        get_all_users_ = get_all_users;
        get_changed_users_ = get_changed_users;
    }

    const std::string& my_node_id() const
    {
        return shards_distributor_->my_node_id();
    }

private:
    void on_acquire_shard(const shard_id& shard)
    {
        YLOG_CTX_LOCAL(context_, info) << "users_distributor_impl::on_acquire_shard " << shard;
        auto listener = std::make_shared<users_listener>(
            io_,
            context_,
            shard,
            get_all_users_,
            get_changed_users_,
            users_polling_settings_,
            io_.wrap(std::bind(
                &users_distributor_impl::on_acquire_uids,
                yplatform::shared_from(this),
                shard,
                ph::_1)),
            io_.wrap(std::bind(
                &users_distributor_impl::on_remove_uids_from_shard,
                yplatform::shared_from(this),
                shard,
                ph::_1)));
        listener->logger(logger());
        users_listeners_[shard] = listener;
        yplatform::spawn(io_.get_executor(), listener);
    }

    void on_acquire_uids(const shard_id& shard, const std::vector<uid_t>& acquired_uids)
    {
        if (!users_listeners_.count(shard))
        {
            YLOG_CTX_LOCAL(context_, info) << "on_acquire_uids: shard not owned, shard=" << shard;
            return;
        }
        std::map<shard_id, std::vector<uid_t>> released_uids;
        for (auto uid : acquired_uids)
        {
            auto it = acquired_users_shard_id_.find(uid);
            if (it != acquired_users_shard_id_.end() && it->second != shard)
            {
                // If uid lived in another shard we release it and acquire again with current shard.
                released_uids[it->second].push_back(uid);
            }
            acquired_users_shard_id_[uid] = shard;
        }
        for (auto& [shard, uids] : released_uids)
        {
            for (auto& cb : on_release_users_)
            {
                io_.post(std::bind(cb, shard, uids));
            }
        }
        for (auto& cb : on_acquire_users_)
        {
            io_.post(std::bind(cb, shard, acquired_uids));
        }
    }

    void on_remove_uids_from_shard(const shard_id& shard, const std::vector<uid_t>& removed_uids)
    {
        // We should check that users still assigned to this shard.
        std::vector<uid_t> uids;
        uids.reserve(removed_uids.size());
        for (auto uid : removed_uids)
        {
            auto it = acquired_users_shard_id_.find(uid);
            if (it != acquired_users_shard_id_.end() && it->second == shard)
            {
                uids.push_back(uid);
                acquired_users_shard_id_.erase(it);
            }
        }
        if (uids.size())
        {
            for (auto& cb : on_release_users_)
            {
                io_.post(std::bind(cb, shard, uids));
            }
        }
    }

    void on_release_shard(const shard_id& shard)
    {
        YLOG_CTX_LOCAL(context_, info) << "users_distributor_impl::on_release_shard " << shard;

        auto listener_it = users_listeners_.find(shard);
        if (listener_it == users_listeners_.end())
        {
            YLOG_CTX_LOCAL(context_, info) << "on_release_shard: users_listener not found";
            return;
        }

        auto& listener = listener_it->second;
        listener->stop();
        users_listeners_.erase(listener_it);

        std::vector<uid_t> uids;
        for (auto it = acquired_users_shard_id_.begin(); it != acquired_users_shard_id_.end();)
        {
            auto& [uid, user_shard_id] = *it;
            if (user_shard_id == shard)
            {
                uids.push_back(uid);
                it = acquired_users_shard_id_.erase(it);
            }
            else
            {
                ++it;
            }
        }
        if (uids.size())
        {
            for (auto& cb : on_release_users_)
            {
                io_.post(std::bind(cb, shard, uids));
            }
        }
    }

    void on_acquire_shards(const std::vector<shard_id>& shards)
    {
        for (auto& shard : shards)
        {
            on_acquire_shard(shard);
        }
    }

    void on_release_shards(const std::vector<shard_id>& shards)
    {
        for (auto& shard : shards)
        {
            on_release_shard(shard);
        }
    }

    std::vector<shard_id_with_uids_cb> on_acquire_users_;
    std::vector<shard_id_with_uids_cb> on_release_users_;
    get_all_users_method get_all_users_;
    get_changed_users_method get_changed_users_;

    boost::asio::io_service& io_;
    task_context_ptr context_;
    ShardsDistributorPtr shards_distributor_;
    SharpeiClientPtr sharpei_client_;
    users_listener::settings users_polling_settings_;
    std::unordered_map<shard_id, users_listener_ptr> users_listeners_;
    std::unordered_map<uid_t, shard_id> acquired_users_shard_id_;
};

}