#pragma once

#include <ymod_mdb_sharder/types.h>

#include <yplatform/coroutine.h>
#include <yplatform/yield.h>
#include <boost/asio/io_service.hpp>
#include <unordered_map>

namespace ymod_mdb_sharder {

// Should be spawned on single thread io_service
struct users_listener
    : std::enable_shared_from_this<users_listener>
    , yplatform::log::contains_logger
{
    using yield_context = yplatform::yield_context<users_listener>;

    struct settings
    {
        time_traits::duration get_changed_users_interval = {};
        time_traits::duration get_all_users_interval = {};
    };

    users_listener(
        boost::asio::io_service& io,
        task_context_ptr ctx,
        const shard_id& shard,
        get_all_users_method get_all_users,
        get_changed_users_method get_changed_users,
        const settings& st,
        const uids_cb& on_acquire,
        const uids_cb& on_release)
        : io(io)
        , context_(ctx)
        , shard(shard)
        , get_all_users(get_all_users)
        , get_changed_users(get_changed_users)
        , st(st)
        , on_acquire(on_acquire)
        , on_release(on_release)
        , timer(io)
    {
    }

    void operator()(yield_context ctx, error_code ec = {}, const shard_users& users = {})
    {
        try
        {
            reenter(ctx)
            {
                while (!stopped)
                {
                    if (time_traits::clock::now() >=
                        last_full_select_ts + st.get_all_users_interval)
                    {
                        yield get_all_users(shard, ctx);
                        last_full_select_ts = time_traits::clock::now();
                    }
                    else
                    {
                        yield get_changed_users(shard, last_moved_ts, last_deleted_ts, ctx);
                    }

                    if (stopped)
                    {
                        break;
                    }

                    if (ec)
                    {
                        YLOG_CTX_LOCAL(context_, error) << "get users error: " << ec.message();
                    }
                    else
                    {
                        update_timestamps(users);
                        notify(users);
                    }

                    yield run_timer(ctx);
                }
            }
        }
        catch (const std::exception& e)
        {
            YLOG_CTX_LOCAL(context_, error) << "users_listener exception: " << e.what();

            // Restart coro.
            run_timer([this, self = shared_from_this()](error_code) {
                yplatform::spawn(io.get_executor(), self);
            });
        }
    }

    void stop()
    {
        io.post([this, self = shared_from_this()] { stopped = true; });
    }

    void notify(const shard_users& users)
    {
        std::vector<uid_t> acquired_uids, released_uids;
        acquired_uids.reserve(users.size());
        released_uids.reserve(users.size());
        for (auto& user : users)
        {
            if (!user.is_here || user.is_deleted)
            {
                released_uids.push_back(user.uid);
            }
            else
            {
                acquired_uids.push_back(user.uid);
            }
        }
        if (acquired_uids.size())
        {
            io.post(std::bind(on_acquire, acquired_uids));
        }
        if (released_uids.size())
        {
            io.post(std::bind(on_release, released_uids));
        }
    }

    template <typename Callback>
    void run_timer(Callback&& cb)
    {
        try
        {
            timer.expires_from_now(st.get_changed_users_interval);
            timer.async_wait(std::forward<Callback>(cb));
        }
        catch (const std::exception& e)
        {
            YLOG_CTX_LOCAL(context_, error)
                << "users_listener exception, application will be aborted: " << e.what();
            abort();
        }
    }

    void update_timestamps(const shard_users& users)
    {
        auto now = std::time(nullptr);
        for (auto& user : users)
        {
            if (!user.is_here && last_moved_ts < user.purge_ts)
            {
                last_moved_ts = user.purge_ts;
            }

            if (user.is_deleted && last_deleted_ts < user.purge_ts)
            {
                last_deleted_ts = user.purge_ts;
            }

            // Check for `here_since_ts <= now` because of existence of users where it far in the
            // future.
            if (user.is_here && !user.is_deleted && last_moved_ts < user.here_since_ts &&
                user.here_since_ts <= now)
            {
                last_moved_ts = user.here_since_ts;
            }
        }
    }

    boost::asio::io_service& io;
    task_context_ptr context_;
    shard_id shard;
    get_all_users_method get_all_users;
    get_changed_users_method get_changed_users;
    settings st;
    uids_cb on_acquire;
    uids_cb on_release;
    time_traits::time_point last_full_select_ts = time_traits::time_point::min();
    time_traits::timer timer;
    std::time_t last_moved_ts = 0;
    std::time_t last_deleted_ts = 0;
    bool stopped = false;
};

using users_listener_ptr = std::shared_ptr<users_listener>;

}

#include <yplatform/unyield.h>