#include "client.h"
#include "client_impl.h"
#include "config.h"
#include "types.h"
#include "resolver.h"

namespace NMds {


std::string MakeStid(const std::string& uid, bool isShared) {
    if (isShared) {
        return "mail:0";
    }
    return "mail:" + uid;
}

void TClient::init(const yplatform::ptree& pt) {
    const auto reactor = pt.get<std::string>("reactor");
    Reactor = yplatform::find_reactor(reactor);

    Cfg.ParsePtree(pt);

    HttpClient = yplatform::find<yhttp::simple_call, std::shared_ptr>(Cfg.HttpClientName);
    if (Cfg.Resolver.Use) {
        HostResolver = MakeHostResolver(*Reactor->io(), Cfg.Resolver, HttpClient);
    }
}

void TClient::start() {
    if (HostResolver) {
        HostResolver->Start();
    }
}

void TClient::stop() {
    if (HostResolver) {
        HostResolver->Stop();
    }
}

void TClient::Put(
    TContext ctx,
    const std::string& id,
    const yplatform::zerocopy::segment& body,
    ENsType nsType,
    TGetPutCallback cb)
{
    auto impl = std::make_shared<TImpl>(ctx, Cfg, HttpClient);
    impl->Put(GetHost(), id, body, nsType, cb);
}

void TClient::Put(
    TContext ctx,
    const std::string& id,
    std::string body,
    ENsType nsType,
    TGetPutCallback cb)
{
    auto impl = std::make_shared<TImpl>(ctx, Cfg, HttpClient);
    impl->Put(GetHost(), id, std::move(body), nsType, std::move(cb));
}

void TClient::Check(TContext ctx, const std::string& stid, TCheckDelCallback cb) {
    auto impl = std::make_shared<TImpl>(ctx, Cfg, HttpClient);
    impl->Check(stid, cb);
}

void TClient::Del(TContext ctx, const std::string& stid, TCheckDelCallback cb) {
    auto impl = std::make_shared<TImpl>(ctx, Cfg, HttpClient);
    impl->Del(stid, cb);
}

void TClient::Get(TContext ctx, const std::string& stid, TGetPutCallback cb) {
    auto impl = std::make_shared<TImpl>(ctx, Cfg, HttpClient);
    impl->Get(stid, cb);
}

std::string TClient::GetHost() const {
    if (HostResolver) {
        return HostResolver->GetRealHost();
    }
    return "";  // means balancer
}

}  // namespace NMds

#include <yplatform/module_registration.h>
REGISTER_MODULE(NMds::TClient)
