#include "client_impl.h"
#include <mail/ymod_tvm/include/ymod_tvm/module.h>

namespace NMds {

void AppendUriParams(std::string& uri, const std::string& paramsStr) {
    if (paramsStr.empty()) {
        return;
    }

    uri.reserve(uri.size() + paramsStr.size() + 1);

    if (uri.rfind('?') == std::string::npos && uri.rfind('&') == std::string::npos) {
        uri.append("?");
    } else if (uri.back() != '?' && uri.back() != '&') {
        uri.append("&");
    }

    uri.append(paramsStr);
}

std::string MakeUrl(
    const std::string& meth,
    const std::string& param,
    const TConfig& cfg)
{
    std::string url{"/"};
    url.append(meth).append("/").append(param);

    AppendUriParams(
        url,
        yhttp::form_encode({
            {"elliptics", 1},
            {"service", cfg.ServiceName}
        }));
    return url;
}

std::string MakeUrl(
    const std::string& meth,
    const std::string& param,
    const TConfig& cfg,
    ENsType nsType)
{
    auto url = MakeUrl(meth, param, cfg);

    if (nsType == NS_TMP) {
        // TODO(MAILDLV-3492)
        AppendUriParams(url, yhttp::form_encode({{"ns", cfg.NamespaceTmp}}));
    } else {
        AppendUriParams(
            url,
            yhttp::form_encode({
                {"unit_type", nsType == NS_SPAM ? "spam" : "ham"},
                {"ns", nsType == NS_SPAM ? cfg.NamespaceSpam : cfg.Namespace}
            }));
    }

    return url;
}

std::string MakeHeaders(const TConfig& cfg) {
    if (!cfg.Tvm.Use) {
        return {};
    }

    auto tvm = yplatform::find<ymod_tvm::tvm2_module>(cfg.Tvm.ModuleName);
    std::string serviceTicket;
    auto err = tvm->get_service_ticket(cfg.Tvm.ServiceName, serviceTicket);
    if (err) {
        throw std::runtime_error("failed to get service ticket for " + cfg.Tvm.ServiceName + ": " + err.message());
    }

    if (serviceTicket.empty()) {
        return {};
    }

    return "X-Ya-Service-Ticket: " + serviceTicket + "\r\n";
}

TImpl::TImpl(TContext ctx, TConfig cfg, THttpClientPtr httpClient)
    : yplatform::log::contains_logger(ctx->logger())
    , Cfg(cfg)
    , Ctx(ctx)
    , HttpClient(httpClient)
{
    logger().set_log_prefix(ctx->uniq_id());
}

void TImpl::Put(
    const std::string& host,
    const std::string& id,
    const yplatform::zerocopy::segment& body,
    ENsType nsType,
    TGetPutCallback cb)
{
    return Put(host, id, {body.begin(), body.end()}, nsType, std::move(cb));
}

void TImpl::Put(
    const std::string& host,
    const std::string& id,
    std::string body,
    ENsType nsType,
    TGetPutCallback cb)
{
    std::string baseUrl = Cfg.Url;

    try {
        if (!host.empty()) {
            baseUrl.assign("http://").append(host);
            if (Cfg.RealPort != 0) {
                baseUrl.append(":").append(std::to_string(Cfg.RealPort));
            }
            baseUrl.append("/gate");
        }
    } catch (const std::exception& e) {
        YLOG_L(error) << "make put url error: " << e.what();
        return LogAndCallback(std::move(cb), EError::StorageError, "");
    }

    auto url = baseUrl + MakeUrl("put", id, Cfg, nsType);
    auto headers = MakeHeaders(Cfg);
    auto self = shared_from_this();

    HttpClient->async_run(Ctx, yhttp::request::POST(url, headers, std::move(body)),
        [this, self, cb = std::move(cb)](auto ec, auto r) {
            OnPut(std::move(cb), ec, r);
        });
}

void TImpl::Check(const std::string& stid, TCheckDelCallback cb) {
    auto url = Cfg.Url + MakeUrl("get", stid, Cfg);
    auto headers = MakeHeaders(Cfg);
    auto self = shared_from_this();

    HttpClient->async_run(Ctx, yhttp::request::HEAD(url, headers),
        [this, self, cb = std::move(cb)](auto ec, auto r) {
            OnCheckDel(std::move(cb), ec, r);
        });
}

void TImpl::Del(const std::string& stid, TCheckDelCallback cb) {
    auto url = Cfg.Url + MakeUrl("del", stid, Cfg);
    auto headers = MakeHeaders(Cfg);
    auto self = shared_from_this();

    HttpClient->async_run(Ctx, yhttp::request::GET(url, headers),
        [this, self, cb = std::move(cb)](auto ec, auto r) {
            OnCheckDel(std::move(cb), ec, r);
        });
}

void TImpl::Get(const std::string& stid, TGetPutCallback cb) {
    auto url = Cfg.Url + MakeUrl("get", stid, Cfg);
    auto headers = MakeHeaders(Cfg);
    auto self = shared_from_this();

    HttpClient->async_run(Ctx, yhttp::request::GET(url, headers),
        [this, self, cb = std::move(cb)](auto ec, auto r) {
            OnGet(std::move(cb), ec, r);
        });
}

void TImpl::OnPut(TGetPutCallback cb, TErrorCode ec, const yhttp::response& r) {
    if (ec) {
        return LogAndCallback(std::move(cb), ec, "");
    }

    if (r.status / 100 != 2) { // 2xx
        ec = EError::StorageError;
    } else {
        YLOG_L(debug) << "put successful, stid=" << r.body << " , mulcaid=" << r.body;
    }

    return LogAndCallback(std::move(cb), ec, r.body);
}

void TImpl::OnGet(TGetPutCallback cb, TErrorCode ec, const yhttp::response& r) {
    if (ec) {
        return LogAndCallback(std::move(cb), ec, "");
    }

    if (r.status == 404) {
        ec = EError::StorageMailNotFound;
    } else if (r.status / 100 != 2) { // 2xx
        ec = EError::StorageError;
    }

    return LogAndCallback(std::move(cb), ec, r.body);
}

void TImpl::OnCheckDel(
    TCheckDelCallback cb,
    TErrorCode ec,
    const yhttp::response& r)
{
    if (ec) {
        return LogAndCallback(std::move(cb), ec, false);
    }

    if (r.status == 404) {
        return LogAndCallback(std::move(cb), ec, false);
    } else if (r.status / 100 != 2) { // 2xx
        ec = EError::StorageError;
        return LogAndCallback(std::move(cb), ec, false);
    }

    return LogAndCallback(std::move(cb), ec, true);
}

}  // namespace NMds
