#pragma once

#include <boost/system/error_code.hpp>

namespace NMds {

using TErrorCode = ::boost::system::error_code;
using TErrorCategory = ::boost::system::error_category;

enum class EError {
    Ok = 0,
    StorageError,
    StorageMailNotFound,
};

const TErrorCategory& category() noexcept;

} // namespace NMds

namespace boost {
namespace system {

template <>
struct is_error_code_enum<::NMds::EError>: public std::true_type {};

} // namespace system
} // namespace boost

namespace NMds {

inline auto make_error_code(const EError e) {
    return boost::system::error_code(static_cast<int>(e), category());
}

class TCategory: public boost::system::error_category {
public:
    const char* name() const noexcept override { return "NMds::TCategory"; }

    std::string message(int value) const override {
        switch (static_cast<EError>(value)) {
            case EError::Ok:
                return "ok";
            case EError::StorageError:
                return "storage error";
            case EError::StorageMailNotFound:
                return "message not found in the storage";
        }

        return "unknown error (" + std::to_string(value) + ")";
    }
};

inline const TErrorCategory& category() noexcept {
    static TCategory instance;
    return instance;
}

} // namespace NMds
