#include "errors.h"
#include "types.h"
#include "resolver.h"

#include <yplatform/coroutine.h>

#include <boost/property_tree/ptree.hpp>
#include <boost/property_tree/json_parser.hpp>

#include <shared_mutex>

namespace NMds {

#include <yplatform/yield.h>
class THostResolver final
    : public IHostResolver
    , public std::enable_shared_from_this<THostResolver> {
public:
    THostResolver(
        boost::asio::io_service& io,
        TConfig::THostResolver config,
        THttpClientPtr httpClient
    )
        : Io(io)
        , Config(config)
        , HttpClient(httpClient) {}

    ~THostResolver() = default;

    void Start() override {
        Timer = std::make_unique<TTimer>(Io);
        yplatform::spawn(shared_from_this());
    }

    void Stop() override {
        TErrorCode ec{};
        Timer->cancel(ec);
    }

    std::string GetRealHost() const override {
        TReadLock lock(Mutex);
        return MulcagateReal;
    }

    using TYieldCtx = yplatform::yield_context<THostResolver>;

    void operator()(
        TYieldCtx yctx,
        TErrorCode ec = TErrorCode{},
        const yhttp::response& response = yhttp::response())
    {
        reenter (yctx) {
            while (true) {
                yield HttpClient->async_run(Ctx, yhttp::request::GET(Config.Url), yctx);

                if (!ec && response.status == 200) {
                    using boost::property_tree::ptree;
                    using boost::property_tree::json_parser::read_json;

                    try {
                        ptree data;
                        std::istringstream is(response.body);
                        read_json(is, data);
                        std::string result;

                        result = data.get<std::string>("host");
                        if (!result.empty()) {
                            TWriteLock lock(Mutex);
                            MulcagateReal.swap(result);
                        }
                    } catch (...) {
                        // ignore
                    }
                }

                yield {
                    try {
                        Timer->expires_from_now(TSeconds(Config.Period));
                        return Timer->async_wait(yctx);
                    } catch (const std::exception& e) {
                        YLOG_G(error) << "mgresolver timer error: " << e.what();
                        abort();
                    }
                }
            }
        }
    }

private:
    using TMutex = std::shared_timed_mutex;
    using TReadLock = std::shared_lock<TMutex>;
    using TWriteLock = std::unique_lock<TMutex>;

    boost::asio::io_service& Io;
    TConfig::THostResolver Config;
    THttpClientPtr HttpClient;

    std::unique_ptr<TTimer> Timer;
    yplatform::task_context_ptr Ctx = boost::make_shared<yplatform::task_context>();

    std::string MulcagateReal;

    mutable TMutex Mutex;
};

#include <yplatform/unyield.h>

std::shared_ptr<IHostResolver> MakeHostResolver(
    boost::asio::io_service& io,
    TConfig::THostResolver config,
    THttpClientPtr httpClient)
{
    return std::make_shared<THostResolver>(io, config, httpClient);
}

} // namespace NMds
