#include "http_client_mock.h"

#include <mail/ymod_mds/src/client_impl.h>

#include <ymod_httpclient/errors.h>
#include <yplatform/zerocopy/streambuf.h>

#include <gtest/gtest.h>
#include <gmock/gmock.h>

using namespace std::literals;
using namespace NMds;

using testing::_;
using testing::A;
using testing::InvokeArgument;

struct TMdsClientTest : public testing::Test {
    void SetUp() override {
        Ctx = boost::make_shared<yplatform::task_context>();
        Cfg = TConfig{
            .Url="mg",
            .NamespaceSpam="mail-spam",
            .NamespaceTmp="mail-tmp",
            .Namespace="mail",
            .ServiceName="srv",
            .Tvm = TConfig::TTvmConfig{
                .Use = false
            }
        };
        Http = std::make_shared<THttpClientMock>();
        Impl = std::make_shared<TImpl>(Ctx, Cfg, Http);

        yplatform::zerocopy::streambuf buf;
        std::ostream os(&buf);
        os << "data" << std::flush;
        ZcBuf.append(buf.detach(buf.end()));
    }

    void PutZC(const std::string& host, const std::string& id, const yplatform::zerocopy::segment& zc, ENsType ns, TGetPutCallback cb) {
        return Impl->Put(host, id, zc, ns, std::move(cb));
    }

    void PutStr(const std::string& host, const std::string& id, std::string body, ENsType ns, TGetPutCallback cb) {
        return Impl->Put(host, id, std::move(body), ns, std::move(cb));
    }

    template <typename TCallback, typename TCallable, typename TThis, typename ... TArgs>
    void ExpectCallbackCalled(TCallback&& cb, int count, TCallable&& call, TThis* self, TArgs&&... args) {
        int called = 0;
        (self->*call)(std::forward<TArgs>(args)...,
            [cb = std::move(cb), &called](auto ... cbArgs) {
                ++called;
                cb(cbArgs...);
            });
        EXPECT_EQ(called, count);
    }

    void TestCheckDel(
        bool isCheck,
        const std::pair<TErrorCode, yhttp::response>& reply,
        TCheckDelCallback check)
    {
        if (isCheck) {
            EXPECT_CALL(*Http, async_run(_, yhttp::request::HEAD("mg/get/"s + StId +
                "?elliptics=1&service=srv"), _, _)).WillOnce(InvokeArgument<3>(reply.first, reply.second));
        } else {
            EXPECT_CALL(*Http, async_run(_, yhttp::request::GET("mg/del/"s + StId +
                "?elliptics=1&service=srv"), _, _)).WillOnce(InvokeArgument<3>(reply.first, reply.second));
        }

        ExpectCallbackCalled(std::move(check), 1, (isCheck ? &TImpl::Check : &TImpl::Del), Impl.get(), StId);
    }

    TContext Ctx;
    std::shared_ptr<THttpClientMock> Http;
    TConfig Cfg;
    std::shared_ptr<TImpl> Impl;
    yplatform::zerocopy::segment ZcBuf;
    const std::string StId{"320.mail:1234.E17:17171717"};
    const std::string PutStId{"mail:1234"};
};

TEST_F(TMdsClientTest, CheckOk) {
    TestCheckDel(true,
        {EError::Ok, {204, {}, "", "No Content"}},
        [](TErrorCode ec, bool found) {
            EXPECT_FALSE(ec);
            EXPECT_TRUE(found);
        });
}

TEST_F(TMdsClientTest, CheckNotFound) {
    TestCheckDel(true,
        {EError::Ok, {404, {}, "", "Not Found"}},
        [](TErrorCode ec, bool found) {
            EXPECT_FALSE(ec);
            EXPECT_FALSE(found);
        });
}

TEST_F(TMdsClientTest, CheckError) {
    TestCheckDel(true,
        {EError::StorageError, {204, {}, "", "Not Found"}},
        [](TErrorCode ec, bool found) {
            EXPECT_TRUE(ec);
            EXPECT_FALSE(found);
        });
}

TEST_F(TMdsClientTest, DelOk) {
    TestCheckDel(false,
        {EError::Ok, {204, {}, "", "No Content"}},
        [](TErrorCode ec, bool found) {
            EXPECT_FALSE(ec);
            EXPECT_TRUE(found);
        });
}

TEST_F(TMdsClientTest, DelNotFound) {
    TestCheckDel(false,
        {EError::Ok, {404, {}, "", "Not Found"}},
        [](TErrorCode ec, bool found) {
            EXPECT_FALSE(ec);
            EXPECT_FALSE(found);
        });
}

TEST_F(TMdsClientTest, DelError) {
    TestCheckDel(false,
        {EError::StorageError, {204, {}, "", "Not Found"}},
        [](TErrorCode ec, bool found) {
            EXPECT_TRUE(ec);
            EXPECT_FALSE(found);
        });
}

TEST_F(TMdsClientTest, PutStringHam) {
    std::string url{"http://mgreal/gate/put/"s + PutStId + "?elliptics=1&service=srv&unit_type=ham&ns=mail"};
    EXPECT_CALL(*Http, async_run(_, yhttp::request::POST(url, "string"), _, _))
        .WillOnce(InvokeArgument<3>(EError::Ok, yhttp::response{201, {}, StId, "Created"}));

    ExpectCallbackCalled(
        [this](TErrorCode ec, const std::string& s) {
            EXPECT_FALSE(ec);
            EXPECT_EQ(s, StId);
        },
        1,
        &TMdsClientTest::PutStr, this, "mgreal", PutStId, "string", NS_MAIL);
}

TEST_F(TMdsClientTest, PutHamToRealNoPort) {
    std::string url{"http://mgreal/gate/put/"s + PutStId + "?elliptics=1&service=srv&unit_type=ham&ns=mail"};
    EXPECT_CALL(*Http, async_run(_, yhttp::request::POST(url, {ZcBuf.begin(), ZcBuf.end()}), _, _))
        .WillOnce(InvokeArgument<3>(EError::Ok, yhttp::response{201, {}, StId, "Created"}));

    ExpectCallbackCalled(
        [this](TErrorCode ec, const std::string& s) {
            EXPECT_FALSE(ec);
            EXPECT_EQ(s, StId);
        },
        1,
        &TMdsClientTest::PutZC, this, "mgreal", PutStId, ZcBuf, NS_MAIL);
}

TEST_F(TMdsClientTest, PutSpamToRealWithPort) {
    Cfg.RealPort = 10010;
    Cfg.NamespaceSpam = "mail-spam";
    Impl = std::make_shared<TImpl>(Ctx, Cfg, Http);

    std::string url{"http://mgreal:10010/gate/put/"s + PutStId + "?elliptics=1&service=srv&unit_type=spam&ns=mail-spam"};
    EXPECT_CALL(*Http, async_run(_, yhttp::request::POST(url, {ZcBuf.begin(), ZcBuf.end()}), _, _))
        .WillOnce(InvokeArgument<3>(EError::Ok, yhttp::response{201, {}, StId, "Created"}));

    ExpectCallbackCalled(
        [this](auto ec, const std::string& got_StId) {
            EXPECT_FALSE(ec);
            EXPECT_EQ(got_StId, StId);
        },
        1,
        &TMdsClientTest::PutZC, this, "mgreal", PutStId, ZcBuf, NS_SPAM);
}

TEST_F(TMdsClientTest, PutHamToEmptyReal) {
    Cfg.RealPort = 10010;
    Impl = std::make_shared<TImpl>(Ctx, Cfg, Http);

    std::string url{"mg/put/"s + PutStId + "?elliptics=1&service=srv&unit_type=ham&ns=mail"};
    EXPECT_CALL(*Http, async_run(_, yhttp::request::POST(url, {ZcBuf.begin(), ZcBuf.end()}), _, _))
        .WillOnce(InvokeArgument<3>(EError::Ok, yhttp::response{201, {}, StId, "Created"}));

    ExpectCallbackCalled(
        [this](TErrorCode ec, const std::string& s) {
            EXPECT_FALSE(ec);
            EXPECT_EQ(s, StId);
        },
        1,
        &TMdsClientTest::PutZC, this, "", PutStId, ZcBuf, NS_MAIL);
}

TEST_F(TMdsClientTest, PutHamFailed) {
    std::string url{"mg/put/"s + PutStId + "?elliptics=1&service=srv&unit_type=ham&ns=mail"};
    EXPECT_CALL(*Http, async_run(_, yhttp::request::POST(url, {ZcBuf.begin(), ZcBuf.end()}), _, _))
        .WillOnce(InvokeArgument<3>(EError::Ok, yhttp::response{500, {}, "", "Internal Server Error"}));

    ExpectCallbackCalled(
        [](TErrorCode ec, const std::string& s) {
            EXPECT_TRUE(ec);
            EXPECT_TRUE(s.empty());
        },
        1,
        &TMdsClientTest::PutZC, this, "", PutStId, ZcBuf, NS_MAIL);
}

TEST_F(TMdsClientTest, PutTmpToEmptyReal) {
    Cfg.RealPort = 10010;
    Impl = std::make_shared<TImpl>(Ctx, Cfg, Http);

    std::string url{"mg/put/"s + PutStId + "?elliptics=1&service=srv&ns=mail-tmp"};
    EXPECT_CALL(*Http, async_run(_, yhttp::request::POST(url, {ZcBuf.begin(), ZcBuf.end()}), _, _))
        .WillOnce(InvokeArgument<3>(EError::Ok, yhttp::response{201, {}, StId, "Created"}));

    ExpectCallbackCalled(
        [this](TErrorCode ec, const std::string& s) {
            EXPECT_FALSE(ec);
            EXPECT_EQ(s, StId);
        },
        1,
        &TMdsClientTest::PutZC, this, "", PutStId, ZcBuf, NS_TMP);
}

// TODO: Add some TVM tests?
