#pragma once

#include <ymod_messenger/detail/types.h>
#include <algorithm>
#include <array>
#include <cassert>
#include <memory>
#include <type_traits>
#include <vector>

namespace ymod_messenger {

/*
 * Continuous buffer types with data copy.
 */
class buffer
{
public:
    buffer() : offset_(0U), size_(0U)
    {
    }

    explicit buffer(std::vector<char>&& buffer)
        : impl_(std::move(buffer)), offset_(0U), size_(impl_.size())
    {
    }

    explicit buffer(std::vector<char>&& buffer, size_t offset, size_t size)
        : impl_(std::move(buffer)), offset_(offset), size_(size)
    {
    }

    const char* data() const
    {
        return impl_.data() + offset_;
    }

    size_t size() const
    {
        return size_;
    }

private:
    std::vector<char> impl_;
    size_t offset_; // offset instead of data pointer allows to use default ctors
    size_t size_;
};

/*
 * Continuous buffer types with shared data.
 */
class shared_buffer
{
public:
    shared_buffer() : data_(nullptr), size_(0)
    {
    }

    explicit shared_buffer(std::shared_ptr<char> buffer, size_t size)
        : impl_(buffer), data_(buffer.get()), size_(size)
    {
    }

    explicit shared_buffer(std::shared_ptr<char> buffer, const char* data_begin, size_t size)
        : impl_(buffer), data_(data_begin), size_(size)
    {
    }

    const char* data() const
    {
        return data_;
    }

    size_t size() const
    {
        return size_;
    }

    std::shared_ptr<char> native() const
    {
        return impl_;
    }

    template <typename Container>
    static shared_buffer create_from(const Container& src)
    {
        std::shared_ptr<char> tmp(new char[src.size()], std::default_delete<char[]>());
        std::copy(src.data(), src.data() + src.size(), tmp.get());
        return shared_buffer(tmp, src.size());
    }

private:
    std::shared_ptr<char> impl_;
    const char* data_;
    size_t size_;
};

/*
 * Shared buffer sequence with limited length.
 */
class shared_buffers
{
    using impl_type = std::array<shared_buffer, detail::MAX_SEND_SEQ_LEN>;

public:
    shared_buffers() : size_(0U)
    {
    }

    shared_buffers(shared_buffer* buffers, size_t size)
        : size_(std::min<size_t>(size, detail::MAX_SEND_SEQ_LEN))
    {
        assert(size <= detail::MAX_SEND_SEQ_LEN);
        std::copy(buffers, buffers + size_, &seq_[0]);
    }

    impl_type::const_iterator begin() const
    {
        return seq_.begin();
    }

    impl_type::const_iterator end() const
    {
        return seq_.begin() + size_;
    }

    const shared_buffer& operator[](size_t index) const
    {
        return seq_[index];
    }

    size_t size() const
    {
        return size_;
    }

private:
    impl_type seq_;
    size_t size_;
};

template <typename...>
struct is_buffer;

template <typename T>
struct is_buffer<T> : public std::false_type
{
};

template <>
struct is_buffer<buffer> : public std::true_type
{
};

template <>
struct is_buffer<shared_buffer> : public std::true_type
{
};

template <bool B>
using bool_constant = std::integral_constant<bool, B>;

template <typename T, typename... Args>
struct is_buffer<T, Args...>
    : public bool_constant<is_buffer<T>::value && is_buffer<Args...>::value>
{
};

}
