#pragma once

#include <ymod_messenger/buffers.h>
#include <ymod_messenger/detail/fragment.h>
#include <ymod_messenger/detail/header.h>
#include <ymod_messenger/detail/types.h>
#include <msgpack.hpp>
#include <vector>

namespace ymod_messenger { namespace detail {

class segment_t
{
    using impl_type = std::array<fragment, MAX_FRAGMENTS>;

public:
    segment_t(fragment&& f) : size_(1)
    {
        seq_[0] = std::move(f);
    }

    segment_t(fragment&& f1, fragment&& f2) : size_(2)
    {
        seq_[0] = std::move(f1);
        seq_[1] = std::move(f2);
    }

    segment_t(fragment&& f1, fragment&& f2, fragment&& f3) : size_(3)
    {
        seq_[0] = std::move(f1);
        seq_[1] = std::move(f2);
        seq_[2] = std::move(f3);
    }

    impl_type::const_iterator begin() const
    {
        return seq_.begin();
    }

    impl_type::const_iterator end() const
    {
        return seq_.begin() + size_;
    }

    shared_buffers to_shared_buffers() const
    {
        shared_buffer buffers[detail::MAX_SEND_SEQ_LEN];
        for (size_t i = 0; i < size_ - 1; ++i)
        {
            buffers[i] = seq_[i + 1].to_shared_buffer();
        }
        return shared_buffers(buffers, size_ - 1);
    }

    size_t size() const
    {
        return size_;
    }

private:
    impl_type seq_;
    size_t size_;
};

template <typename... Buffers>
inline auto create_segment(message_type type, Buffers&&... buffers) ->
    typename std::enable_if<sizeof...(Buffers) <= MAX_SEND_SEQ_LEN, segment_t>::type
{
    std::vector<char> header_buffer(header_size(sizeof...(Buffers)));
    pack_header(type, buffers.size()..., header_buffer.data(), header_buffer.size());
    return segment_t(
        fragment(buffer(std::move(header_buffer))), fragment(std::forward<Buffers>(buffers))...);
}

template <typename Message>
inline shared_buffer create_msgpack_buffer(Message&& message)
{
    static const size_t DEFAULT_MESSAGE_SIZE = 4 * 1024;
    msgpack::sbuffer buf(DEFAULT_MESSAGE_SIZE);
    msgpack::packer<msgpack::sbuffer> packer(buf);
    packer.pack(message);
    size_t buf_size = buf.size();
    return shared_buffer(std::shared_ptr<char>(buf.release(), free), buf_size);
}

template <typename Message>
inline segment_t create_msgpack_segment(message_type type, Message&& message)
{
    return create_segment(type, create_msgpack_buffer(std::forward<Message>(message)));
}

} // detail
} // ymod_messenger
