#pragma once

#include <ymod_messenger/buffers.h>
#include <ymod_messenger/types.h>
#include <ymod_messenger/detail/message_handler_wrapper.h>
#include <yplatform/module.h>
#include <yplatform/reactor.h>

namespace ymod_messenger {

/*
 * Virtual methods usually are being overloaded in unit tests.
 */
class module : public yplatform::module
{
public:
    module(yplatform::reactor& reactor, const yplatform::ptree& ptree);
    module(boost::asio::io_service& io, const yplatform::ptree& ptree);

    // Start/stop accepting incoming connections.
    virtual void start();
    virtual void stop();

    // Shadows base class method.
    void logger(const yplatform::log::source&);

    // TODO ?typedef address, message, etc?
    virtual const address_t& my_address() const;
    virtual void connect(const string& address);
    virtual void disconnect(const string& address);
    virtual void connect_to_cluster(const std::set<address_t>& peers);

    template <typename MessageType, typename Handler>
    hook_id_t bind_messages(Handler&& handler, const message_type type = message_type_NONE)
    {
        return bind_messages(
            detail::message_handler_wrapper<MessageType, Handler>(std::forward<Handler>(handler)),
            type);
    }

    virtual hook_id_t bind_messages(
        const message_hook_t& hook,
        const message_type type = message_type_NONE);

    virtual hook_id_t bind_events(const event_hook_t& hook);

    template <typename Message>
    void send(const string& address, Message&& message, const message_type type = message_type_NONE)
    {
        segment_t seg = detail::create_msgpack_segment(type, std::forward<Message>(message));
        do_send(address, std::move(seg), type);
    }

    template <typename Message>
    void send_all(
        pool_type_t pool_type,
        Message&& message,
        const message_type type = message_type_NONE)
    {
        segment_t seg = detail::create_msgpack_segment(type, std::forward<Message>(message));
        do_send_all(pool_type, std::move(seg), type);
    }

    template <typename Message>
    void send_all(Message&& message, const message_type type = message_type_NONE)
    {
        send_all(pool_ANY, std::forward<Message>(message), type);
    }

    void send(
        const string& address,
        shared_buffer buffer,
        const message_type type = message_type_NONE)
    {
        segment_t seg = detail::create_segment(type, std::move(buffer));
        do_send(address, std::move(seg), type);
    }

    void send(
        const string& address,
        shared_buffer buffer1,
        shared_buffer buffer2,
        const message_type type = message_type_NONE)
    {
        segment_t seg = detail::create_segment(type, std::move(buffer1), std::move(buffer2));
        do_send(address, std::move(seg), type);
    }

    void send_all(
        pool_type_t pool_type,
        shared_buffer buffer,
        const message_type type = message_type_NONE)
    {
        segment_t seg = detail::create_segment(type, std::move(buffer));
        do_send_all(pool_type, std::move(seg), type);
    }

    void send_all(
        pool_type_t pool_type,
        shared_buffer buffer1,
        shared_buffer buffer2,
        const message_type type = message_type_NONE)
    {
        segment_t seg = detail::create_segment(type, std::move(buffer1), std::move(buffer2));
        do_send_all(pool_type, std::move(seg), type);
    }

    const yplatform::module_stats_ptr get_module_stats() const override;

    virtual ~module()
    {
    }

protected:
    module()
    { /* do nothing, fake ctor for inheritance in tests */
    }

private:
    virtual void do_send(
        const string& address,
        segment_t seg,
        const message_type type = message_type_NONE);
    virtual void do_send_all(
        pool_type_t pool_type,
        segment_t seg,
        const message_type type = message_type_NONE);

    class impl;
    std::shared_ptr<impl> impl_;
};

}
