#pragma once

#include <yplatform/net/module.h>
#include <yplatform/zerocopy/streambuf.h>
#include <yplatform/zerocopy/streambuf.h>
#include <yplatform/module.h>
#include <yplatform/ptree.h>

#include <ymod_messenger/types.h>
#include <ymod_messenger/host_info.h>
#include <ymod_messenger/ymod_messenger.h>
#include "session/client_session.h"
#include "session/server_session.h"
#include "settings.h"
#include "session_factory.h"
#include "pool_manager.h"
#include "outgoing_pool.h"
#include "incoming_pool.h"
#include "stats.h"

namespace ymod_messenger {

using yplatform::ptree;

/**
 * network module
 * - both server and client role
 *   with same session logic and client connections pool support
 */
class module::impl
    : public yplatform::log::contains_logger
    , public std::enable_shared_from_this<module::impl>
{
    typedef yplatform::net::dns::resolver resolver_t;
    typedef yplatform::net::client_settings::resolve_order_t resolve_order_t;
    typedef yplatform::net::server_module<server_session> server_module;
    typedef yplatform::net::client_module<client_session> client_module;
    typedef pool_manager<outgoing_pool, incoming_pool, resolver_t, resolve_order_t> pool_manager_t;
    typedef boost::mutex mutex_t;
    typedef boost::unique_lock<mutex_t> lock_t;

public:
    impl()
        : events_notifier_(new events_notifier()), send_all_with_myself_(true), stats_(new stats())
    {
    }

    void init(yplatform::reactor& reactor, const ptree& config)
    {
        char host_name[256];
        gethostname(host_name, sizeof(host_name));
        my_address_ = host_name;

        server_settings_.logger(logger());
        client_settings_.logger(logger());
        server_settings_.deps_injection.on_server_connection =
            boost::bind(&impl::on_server_connection, this, _1);
        // init settings injection
        if (config.count("endpoints"))
        {
            server_ = make_shared<server_module>("ymod_messenger_server");
            if (!server_->open_server(reactor, config, &server_settings_, logger()))
            {
                throw std::runtime_error("can't init network server");
            }
            assert(server_->local_endpoints().size() != 0);
            unsigned my_port = server_->local_endpoints().front().second;
            my_address_.append(":");
            my_address_.append(boost::lexical_cast<string>(my_port));
        }
        client_ = make_shared<client_module>("ymod_messenger_client");
        if (!client_->open_client(reactor, config, &client_settings_, logger()))
        {
            throw std::runtime_error("can't init network client");
        }

        notifier_.reset(
            new messages_notifier([this]() -> boost::asio::io_service* { return client_->io(); }));

        notifier_->logger(logger());
        events_notifier_->logger(logger());

        time_traits::duration resolve_timeout =
            time_traits::seconds(config.get("resolve_timeout", 3));
        send_all_with_myself_ = config.get("send_all_with_myself", "0") != "0";

        auto resolver = std::make_shared<pool_resolver<resolver_t, resolve_order_t>>(
            *io(),
            resolver_t(*io()),
            resolver_t(*io()),
            client_settings_.resolve_order,
            resolve_timeout);
        resolver->logger(logger());

        session_factory_.reset(new session_factory<client_module>(client_));

        pool_settings pool_st;
        pool_st.logger(logger());
        pool_st.parse_ptree(config);
        pool_manager_.reset(new pool_manager_t(
            resolver, session_factory_, notifier_, events_notifier_, pool_st, stats_, io()));
        pool_manager_->logger(this->logger());

        io_ = reactor.io();

        YLOG_L(debug) << "init server_ use_count: " << server_.use_count();
        YLOG_L(debug) << "init client_ use_count: " << client_.use_count();
        YLOG_L(debug) << "init this use_count: " << shared_from_this().use_count();

        if (config.count("peers"))
        {
            std::set<address_t> peers;
            yplatform::read_ptree(peers, config, "peers");
            connect_to_cluster(peers);
        }
    }

    void start()
    {
        if (server_) server_->run_server();
        client_->run_client();
    }

    void stop()
    {
        if (server_) server_->stop_server();
        client_->stop_client();
    }

    void fini()
    {
        pool_manager_->close();
        pool_manager_.reset();

        if (server_) server_->abort_server();
        // XXX ugly hack to prevent yplatform::net::server being destroyed
        // before it finish all background activities.
        // TODO use net::tcp_server instead.
        io_->post([server = server_]() {});
        server_.reset();
        client_.reset();
    }

    boost::asio::io_service* io()
    {
        return client_->io();
    }

    const address_t& my_address() const
    {
        return my_address_;
    }

    void connect(const string& address)
    {
        if (address == my_address_)
        {
            YLOG_L(warning) << "ignoring attemp to connect to ymod_messenger's address";
            return;
        }

        lock_t lock(mux_);
        if (connected_peers_.count(address))
        {
            YLOG_L(warning) << "endpoint already connected: address=\"" << address << "\"";
            return;
        }
        connected_peers_.insert(address);
        lock.unlock();

        host_info address_host_info = make_host_info(address);
        pool_manager_->open_pool(address_host_info);
    }

    void disconnect(const string& address)
    {
        lock_t lock(mux_);
        if (!connected_peers_.count(address))
        {
            YLOG_L(warning) << "failed to disconnect: endpoint is not connected, address=\""
                            << address << "\"";
            return;
        }
        connected_peers_.erase(address);
        lock.unlock();

        host_info address_host_info = make_host_info(address);
        pool_manager_->close_pool(address_host_info);
    }

    void connect_to_cluster(const std::set<address_t>& peers)
    {
        lock_t lock(mux_);
        std::vector<address_t> removed_peers, new_peers;
        std::set_difference(
            connected_peers_.begin(),
            connected_peers_.end(),
            peers.begin(),
            peers.end(),
            std::back_inserter(removed_peers));
        std::set_difference(
            peers.begin(),
            peers.end(),
            connected_peers_.begin(),
            connected_peers_.end(),
            std::back_inserter(new_peers));

        connected_peers_ = peers;
        lock.unlock();

        for (auto addr_it = removed_peers.begin(); addr_it != removed_peers.end(); addr_it++)
        {
            if (my_address_ == *addr_it) continue;
            YLOG_L(notice) << "disconnecting peer: addr=\"" << *addr_it << "\"";
            host_info address_host_info = make_host_info(*addr_it);
            pool_manager_->close_pool(address_host_info);
        }

        for (auto addr_it = new_peers.begin(); addr_it != new_peers.end(); addr_it++)
        {
            if (my_address_ == *addr_it) continue;
            YLOG_L(notice) << "connecting to peer: addr=\"" << *addr_it << "\"";
            host_info address_host_info = make_host_info(*addr_it);
            pool_manager_->open_pool(address_host_info);
        }
    }

    hook_id_t bind_messages(const message_hook_t& hook, const message_type type)
    {
        return notifier_->add_hook(hook, type);
    }

    hook_id_t bind_events(const event_hook_t& hook)
    {
        return events_notifier_->add_hook(hook);
    }

    void do_send(const string& address, segment_t seg, const message_type type)
    {
        if (address == my_address_)
        {
            send_to_myself(type, seg.to_shared_buffers());
        }
        else
        {
            pool_manager_->send(make_host_info(address), std::move(seg));
        }
    }

    void do_send_all(
        pool_type_t pool_type,
        segment_t seg,
        const message_type type = message_type_NONE)
    {
        if (send_all_with_myself_)
        {
            send_to_myself(type, seg.to_shared_buffers());
        }
        pool_manager_->send_all(pool_type, std::move(seg));
    }

    const yplatform::module_stats_ptr get_module_stats() const
    {
        return stats_;
    }

private:
    void send_to_myself(message_type type, shared_buffers seq) const
    {
        notifier_->notify(my_address_, type, seq);
    }

    void on_server_connection(shared_ptr<server_session> session)
    {
        pool_manager_->add_incoming_session(
            session, host_info(session->remote_addr().to_string(), session->remote_port()));
    }

    shared_ptr<server_module> server_;
    server_settings server_settings_;
    shared_ptr<client_module> client_;
    client_settings client_settings_;

    shared_ptr<session_factory<client_module>> session_factory_;
    shared_ptr<pool_manager_t> pool_manager_;

    shared_ptr<messages_notifier> notifier_;
    shared_ptr<events_notifier> events_notifier_;
    address_t my_address_;
    bool send_all_with_myself_;
    std::set<string> connected_peers_;
    mutex_t mux_;
    stats_ptr stats_;
    boost::asio::io_service* io_ = nullptr;
};

}
