#pragma once

#include <ymod_messenger/types.h>
#include <ymod_messenger/host_info.h>
#include "session/messenger_session.h"
#include "weak_helpers.h"
#include "sessions_selector.h"
#include "pool_base.h"

namespace ymod_messenger {

class incoming_pool
    : public pool_base
    , public boost::enable_shared_from_this<incoming_pool>
{
public:
    incoming_pool(const host_info& addr, stats_ptr stats) : pool_base(addr, stats)
    {
        YLOG_L(debug) << "incoming pool " << address().to_string() << " created";
    }

    ~incoming_pool()
    {
        detach();
        YLOG_L(debug) << "incoming pool " << address().to_string() << " destroyed";
    }

    shared_ptr<incoming_pool> weak_from_this()
    {
        return this->shared_from_this();
    }

    void open()
    {
        lock_t lock(mutex_);
        if (detached_) return;
        if (opened_) return;
        opened_ = true;
    }

    void send(segment_t seg)
    {
        lock_t lock(mutex_);
        if (detached_) return;
        session_ptr session = get_session();
        if (!session)
        {
            YLOG_L(error) << "incoming pool " << address().to_string()
                          << " send failed: no sessions are available";
            return;
        }
        if (lock) lock.unlock();
        session->send(std::move(seg));
    }

    void add_incoming_session(session_ptr session)
    {
        lock_t lock(mutex_);
        if (detached_) return;
        auto add_error = add_session_nolock(session, lock);
        if (lock) lock.unlock();
        if (add_error.empty())
        {
            session->start_read();
        }
        else
        {
            YLOG_L(error) << "incoming pool " << address().to_string()
                          << " add_session failed: " << add_error;
        }
    }

private:
    string add_session_nolock(session_ptr session, lock_t& lock)
    {
        assert(lock);

        if (has(session)) return "session " + session->get_description() + " already in pool";

        session->set_message_hook(
            boost::bind(&incoming_pool::on_message, this->weak_from_this(), _1, _2, _3));
        session->set_error_hook(
            boost::bind(&incoming_pool::on_session_error, this->weak_from_this(), _1, _2));

        if (!session->is_open()) return "session " + session->get_description() + " is not open";

        items_.push_back(session);

        stats_->add_session_stats(address().to_string(), session->get_stats(), pool_INCOMING);
        notify_on_just_opened(lock);

        return "";
    }

    void on_message(session_ptr /*session*/, message_type type, const shared_buffers& seq)
    {
        lock_t lock(mutex_);
        if (detached_) return;

        notify_message(type, seq, lock);
    }

    void on_session_error(session_ptr session, const error_code& err)
    {
        stats_->del_session_stats(session->get_stats());

        lock_t lock(mutex_);
        if (detached_) return;

        if (err)
        {
            YLOG_L(debug) << "incoming_pool " << address().to_string()
                          << " on_session_error: " << err.message();
        }
        for (items_t::iterator it = items_.begin(), end = items_.end(); it != end; ++it)
        {
            if (*it == session)
            {
                items_.erase(it);
                YLOG_L(debug) << "incoming_pool " << address_.to_string()
                              << " connection lost, pool size=" << items_.size();
                if (items_.size() == 0)
                {
                    notify_closed(lock);
                }
                return;
            }
        }
        YLOG_L(warning) << "incoming_pool " << address().to_string()
                        << " on_session_error: no such session";
    }
};

}
