#pragma once

#include <boost/enable_shared_from_this.hpp>
#include <ymod_messenger/types.h>
#include <ymod_messenger/host_info.h>

#include "session/client_session.h"

namespace ymod_messenger {

template <typename HookRecordType>
class hooks_container : public yplatform::log::contains_logger
{
protected:
    typedef HookRecordType hook_record_t;
    typedef std::pair<hook_id_t, hook_record_t> id_hook_pair;
    typedef std::vector<id_hook_pair> hooks_t;
    typedef boost::mutex mutex_t;
    typedef boost::unique_lock<mutex_t> lock_t;

    hooks_container() : id_(0)
    {
    }
    virtual ~hooks_container()
    {
    }

    hook_id_t insert(const hook_record_t& record)
    {
        lock_t lock(mux_);
        ++id_;
        hooks_.push_back(id_hook_pair(id_, record));
        return id_;
    }

    hook_id_t id_;
    hooks_t hooks_;
    mutex_t mux_;
};

struct message_hook_info
{
    message_type type;
    message_hook_t hook;
};

struct event_hook_info
{
    event_hook_t hook;
};

class messages_notifier : public hooks_container<message_hook_info>
{
    typedef message_hook_t hook_t;
    typedef std::function<boost::asio::io_service*()> io_getter_type;

    io_getter_type io_getter_;

public:
    messages_notifier()
    {
    }

    messages_notifier(io_getter_type getter) : io_getter_(getter)
    {
    }

    virtual hook_id_t add_hook(const hook_t& hook, message_type type)
    {
        message_hook_info info = { type, hook };
        return insert(info);
    }

    virtual void notify(const address_t& from, message_type type, const shared_buffers& buffers)
    {
        typedef hooks_t::const_iterator const_iterator;

        hooks_t hooks_copy;
        {
            lock_t lock(mux_);
            hooks_copy = hooks_;
        }
        for (const_iterator it = hooks_copy.begin(), end = hooks_copy.end(); it != end; ++it)
        {
            if (it->second.type == message_type_NONE || it->second.type == type)
            {
                if (io_getter_)
                {
                    // TODO post once in session_base?
                    notify_in_io(it->second.hook, from, type, buffers);
                }
                else
                {
                    it->second.hook(from, buffers);
                }
            }
        }
    }

private:
    void notify_in_io(
        const hook_t& user_hook,
        const address_t& from,
        message_type type,
        const shared_buffers& buffers)
    {
        auto module_logger = this->logger();
        io_getter_()->post([module_logger, user_hook, from, type, buffers]() mutable {
            try
            {
                user_hook(from, buffers);
            }
            catch (const std::exception& e)
            {
                YLOG(module_logger, error) << "messages_notifier hook call exception: " << e.what()
                                           << " message type: " << type;
            }
        });
    }
};

// TODO post to io
class events_notifier : public hooks_container<event_hook_info>
{
    typedef event_notification notification_t;
    typedef event_hook_t hook_t;

public:
    virtual hook_id_t add_hook(const hook_t& hook)
    {
        event_hook_info info = { hook };
        return insert(info);
    }

    virtual void notify(const address_t& from, const notification_t& notification)
    {
        typedef hooks_t::const_iterator const_iterator;

        hooks_t hooks_copy;
        {
            lock_t lock(mux_);
            hooks_copy = hooks_;
        }
        for (const_iterator it = hooks_copy.begin(), end = hooks_copy.end(); it != end; ++it)
        {
            try
            {
                it->second.hook(from, notification);
            }
            catch (std::exception& e)
            {
                YLOG_L(error) << "events_notifier hook call exception: " << e.what();
            }
        }
    }
};

}
