#pragma once

#include <ymod_messenger/types.h>
#include <ymod_messenger/host_info.h>
#include "session_factory.h"
#include "session/messenger_session.h"
#include "session_factory.h"
#include "weak_helpers.h"
#include "sessions_selector.h"
#include "pool_base.h"

namespace ymod_messenger {

class outgoing_pool
    : public pool_base
    , public boost::enable_shared_from_this<outgoing_pool>
{
public:
    typedef pool_base::session_open_hook_t session_open_hook_t;
    typedef pool_base::session_close_hook_t session_close_hook_t;
    typedef pool_base::message_hook_t message_hook_t;

    outgoing_pool(
        const host_info& addr,
        session_factory_ptr factory,
        const pool_settings& settings,
        stats_ptr stats)
        : pool_base(addr, stats), factory_(factory), settings_(settings), opening_sessions_(0)
    {
        YLOG_L(debug) << "outgoing_pool " << address().to_string() << " created";
    }

    ~outgoing_pool()
    {
        detach();
        YLOG_L(debug) << "outgoing_pool " << address().to_string() << " destroyed";
    }

    shared_ptr<outgoing_pool> weak_from_this()
    {
        return this->shared_from_this();
    }

    void open()
    {
        lock_t lock(mutex_);
        if (detached_) return;
        if (opened_) return;
        opened_ = true;

        create_session(time_traits::duration::min(), lock);
    }

    bool is_empty()
    {
        lock_t lock(mutex_);
        return items_.empty() && opening_sessions_ == 0;
    }

    virtual size_t size()
    {
        lock_t lock(mutex_);
        return items_.size() + opening_sessions_;
    }

    void send(segment_t seg)
    {
        lock_t lock(mutex_);
        if (detached_) return;
        session_ptr session = get_session();
        if (!session)
        {
            YLOG_L(error) << "outgoing_pool " << address_.to_string()
                          << " send failed: no sessions are available";
            return;
        }
        if (session->send_queue_size() > 0 && can_create_session())
            create_session(time_traits::duration::min(), lock);
        if (lock) lock.unlock();
        session->send(std::move(seg));
    }

private:
    string add_session_nolock(session_ptr session, lock_t& lock)
    {
        assert(lock);

        if (has(session)) return "session " + session->get_description() + " already in pool";

        session->set_message_hook(
            boost::bind(&outgoing_pool::on_message, this->weak_from_this(), _1, _2, _3));
        session->set_error_hook(
            boost::bind(&outgoing_pool::on_session_error, this->weak_from_this(), _1, _2));

        if (!session->is_open()) return "session " + session->get_description() + " is not open";

        items_.push_back(session);

        stats_->add_session_stats(address().to_string(), session->get_stats(), pool_OUTGOING);
        notify_on_just_opened(lock);

        return "";
    }

    void create_session(const time_traits::duration& delay, lock_t& lock)
    {
        assert(lock);
        if (!opened_)
            throw std::logic_error("failed to create session: outgoing_pool is not opened");
        if (detached_)
            throw std::logic_error("failed to create session: outgoing_pool is detached");
        opening_sessions_++;
        lock.unlock();
        try
        {
            factory_->create_session(
                address_,
                delay,
                boost::bind(&outgoing_pool::on_connected, this->weak_from_this(), _1, _2),
                boost::bind(&outgoing_pool::on_connect_timeout, this->weak_from_this()));
        }
        catch (std::exception& e)
        {
            lock.lock();
            opening_sessions_--;
            throw std::runtime_error(
                string("failed to create session: factory exception ") + e.what());
        }
    }

    void on_connected(session_ptr session, const error_code& err)
    {
        lock_t lock(mutex_);
        opening_sessions_--;
        if (detached_) return;
        if (!opened_) return;

        if (!err)
        {
            YLOG_L(debug) << "outgoing_pool " << address_.to_string()
                          << " connection established, pool size=" << items_.size() + 1;
            auto add_error = add_session_nolock(session, lock);
            if (lock) lock.unlock();
            if (add_error.empty())
            {
                session->start_read();
            }
            else
            {
                YLOG_L(error) << "outgoing_pool " << address_.to_string()
                              << " add_session failed: " << add_error;
            }
        }
        else
        {
            lock.unlock();
            //            YLOG_L(error) << "outgoing_pool " << address_.to_string () << " connect
            //            failed: " << err.message ();
        }

        if (!lock) lock.lock();
        if (can_min_create_session())
        {
            create_session(err ? settings_.reconnect_delay : time_traits::duration::min(), lock);
        }
    }

    bool can_create_session()
    {
        return opened_ && items_.size() + opening_sessions_ < settings_.max_size;
    }

    bool can_min_create_session()
    {
        return opened_ && items_.size() + opening_sessions_ < settings_.min_size;
    }

    void on_connect_timeout()
    {
        lock_t lock(mutex_);
        opening_sessions_--;
        if (detached_) return;
        if (!opened_) return;

        YLOG_L(error) << "outgoing_pool " << address_.to_string() << " connect timeout";
        if (can_create_session())
        {
            create_session(settings_.reconnect_delay, lock);
        }
    }

    void on_message(session_ptr /*session*/, message_type type, const shared_buffers& seq)
    {
        lock_t lock(mutex_);
        if (detached_) return;
        notify_message(type, seq, lock);
    }

    void on_session_error(session_ptr session, const error_code& err)
    {
        stats_->del_session_stats(session->get_stats());

        lock_t lock(mutex_);
        if (detached_) return;

        if (err)
        {
            YLOG_L(debug) << "outgoing_pool " << address().to_string()
                          << " on_session_error: " << err.message();
        }

        for (items_t::iterator it = items_.begin(), end = items_.end(); it != end; ++it)
        {
            if ((*it) == session)
            {
                bool need_reconnect = opened_;
                items_.erase(it);
                // items' size should be checked before create_session call
                bool should_notify_now = items_.size() == 0;
                if (need_reconnect && can_min_create_session())
                {
                    create_session(settings_.reconnect_delay, lock);
                }

                YLOG_L(debug) << "outgoing_pool " << address_.to_string()
                              << " connection lost, pool size=" << items_.size();
                if (!lock) lock.lock();

                if (should_notify_now)
                {
                    notify_closed(lock);
                }
                return;
            }
        }
        YLOG_L(warning) << "outgoing_pool " << address().to_string()
                        << " on_session_error: no such session";
    }

    const session_factory_ptr factory_;
    pool_settings settings_;
    unsigned opening_sessions_;
};

}
