#pragma once

#include <ymod_messenger/types.h>
#include <ymod_messenger/host_info.h>
#include <yplatform/log.h>
#include "session/messenger_session.h"
#include "sessions_selector.h"
#include "stats.h"

namespace ymod_messenger {

class pool_base
    : public sessions_selector
    , public yplatform::log::contains_logger
{
protected:
    typedef messenger_session session_t;
    typedef messenger_session_ptr session_ptr;
    typedef boost::mutex mutex_t;
    typedef boost::unique_lock<mutex_t> lock_t;
    typedef boost::function<void(const host_info&)> session_open_hook_t;
    typedef boost::function<void(const host_info&)> session_close_hook_t;
    typedef boost::function<void(const host_info&, message_type, const shared_buffers&)>
        message_hook_t;

public:
    pool_base(const host_info& address, stats_ptr stats)
        : address_(address), opened_(false), detached_(false), stats_(stats)
    {
    }

    virtual ~pool_base()
    {
    }

    void set_hooks(
        const session_open_hook_t& open_cb,
        const session_close_hook_t& close_cb,
        const message_hook_t& message_cb);

    void close();

    void detach();

    virtual size_t size()
    {
        lock_t lock(mutex_);
        return items_.size();
    }

    virtual bool is_empty()
    {
        lock_t lock(mutex_);
        return items_.empty();
    }

    size_t real_size()
    {
        lock_t lock(mutex_);
        return items_.size();
    }

    virtual void open() = 0;
    virtual void send(segment_t seg) = 0;

    const host_info& address()
    {
        return address_;
    }

protected:
    void notify_on_just_opened(lock_t& lock)
    {
        if (items_.size() == 1)
        {
            notify_opened(lock);
        }
    }

    void notify_opened(lock_t& lock)
    {
        session_open_hook_t hook = hooks_.open;
        lock.unlock();
        if (hook) hook(address_);
    }

    void notify_message(message_type type, const shared_buffers& seq, lock_t& lock)
    {
        message_hook_t hook = hooks_.message;
        lock.unlock();
        if (hook) hook(address_, type, seq);
    }

    void notify_closed(lock_t& lock)
    {
        session_close_hook_t hook = hooks_.close;
        lock.unlock();
        if (hook) hook(address_);
    }

    const host_info address_;
    bool opened_;
    bool detached_;

    struct
    {
        session_open_hook_t open;
        session_close_hook_t close;
        message_hook_t message;

        void clear()
        {
            open.clear();
            close.clear();
            message.clear();
        }
    } hooks_;

    mutex_t mutex_;
    stats_ptr stats_;
};

}
