#pragma once

#include <ymod_messenger/types.h>
#include <yplatform/log.h>
#include "notifier.h"
#include "pool_resolver.h"
#include "weak_helpers.h"
#include "session_factory.h"
#include "stats.h"
#include "timer.h"

namespace ymod_messenger {

// todo: ? optional re-send messages after connection fail ?

template <typename OutgoingPool, typename IncomingPool, typename Resolver, typename ResolveOrder>
class pool_manager
    : public boost::enable_shared_from_this<
          pool_manager<OutgoingPool, IncomingPool, Resolver, ResolveOrder>>
    , public yplatform::log::contains_logger
{
    typedef boost::mutex mutex_t;
    typedef boost::unique_lock<mutex_t> lock_t;
    typedef IncomingPool incoming_pool_t;
    typedef OutgoingPool outgoing_pool_t;
    typedef pool_resolver<Resolver, ResolveOrder> pool_resolver_t;
    typedef shared_ptr<incoming_pool_t> incoming_pool_ptr;
    typedef shared_ptr<outgoing_pool_t> outgoing_pool_ptr;
    typedef std::map<host_info, incoming_pool_ptr> incoming_map_t;
    typedef std::map<host_info, outgoing_pool_ptr> outgoing_map_t;

    typedef boost::function<void(const host_info&)> resolve_hook_t;
    typedef boost::unordered_map<string, string> address_cache_t;

public:
    pool_manager(
        std::shared_ptr<pool_resolver_t> pool_resolver,
        session_factory_ptr session_factory,
        const shared_ptr<messages_notifier> notifier,
        const shared_ptr<events_notifier> events_notifier,
        const pool_settings& settings,
        stats_ptr stats,
        boost::asio::io_service* io = nullptr)
        : pool_resolver_(std::move(pool_resolver))
        , session_factory_(session_factory)
        , notifier_(notifier)
        , events_notifier_(events_notifier)
        , pool_settings_(settings)
        , closed_(false)
        , stats_(stats)
        , io_(io)
    {
    }

    ~pool_manager()
    {
        YLOG_L(debug) << "pool_manager was destroyed";
        if (!is_closed()) close();
        //        clear ();
    }

    void close()
    {
        YLOG_L(debug) << "close()";

        lock_t lock(mux_);
        closed_ = true;

        pool_resolver_->cancel();

        // detach and remove all items
        for (typename outgoing_map_t::iterator it = outgoing_.begin(), end = outgoing_.end();
             it != end;
             ++it)
        {
            it->second->close();
        }
        for (typename incoming_map_t::iterator it = incoming_.begin(), end = incoming_.end();
             it != end;
             ++it)
        {
            it->second->close();
        }

        // detach and remove all items
        for (typename outgoing_map_t::iterator it = outgoing_.begin(), end = outgoing_.end();
             it != end;
             ++it)
        {
            it->second->detach();
        }
        for (typename incoming_map_t::iterator it = incoming_.begin(), end = incoming_.end();
             it != end;
             ++it)
        {
            it->second->detach();
        }
        outgoing_.clear();
        incoming_.clear();
    }

    bool is_closed()
    {
        lock_t lock(mux_);
        return closed_;
    }

    size_t size()
    {
        lock_t lock(mux_);
        return outgoing_.size();
    }

    void add_incoming_session(messenger_session_ptr session, const host_info& remote_address)
    {
        if (is_closed()) return;
        pool_resolver_->resolve_incoming(
            remote_address,
            boost::bind(
                &pool_manager::on_resolve_back_incoming, this->shared_from_this(), _1, session),
            delayed_retry([this, session, remote_address] {
                add_incoming_session(session, remote_address);
            }));
    }

    void open_pool(const host_info& address)
    {
        if (is_closed()) return;
        lock_t lock(mux_);
        outgoing_pool_ptr pool = find_or_create_nolock(address);
        pool->open();
    }

    void close_pool(const host_info& address)
    {
        if (is_closed()) return;
        typename outgoing_map_t::iterator it_pool = outgoing_.find(address);
        lock_t lock(mux_);
        if (it_pool != outgoing_.end())
        {
            it_pool->second->close();
            outgoing_.erase(it_pool);
        }
    }

    void send(const host_info& address, segment_t seg)
    {
        if (is_closed()) return;
        send_resolved(address, std::move(seg));
    }

    size_t send_all(pool_type_t pool_type, const segment_t& seg)
    {
        if (is_closed()) return 0;
        if (pool_type == pool_ANY)
        {
            size_t cnt = send_all(pool_OUTGOING, seg);
            return cnt > 0 ? cnt : send_all(pool_INCOMING, seg);
        }
        else if (pool_type == pool_INCOMING)
        {
            lock_t lock(mux_);
            incoming_map_t conns = incoming_;
            lock.unlock();
            return send_all_in(conns, seg);
        }
        else if (pool_type == pool_OUTGOING)
        {
            lock_t lock(mux_);
            outgoing_map_t conns = outgoing_;
            lock.unlock();
            return send_all_in(conns, seg);
        }

        throw std::runtime_error("Unknown pool_type value");
    }

private:
    template <typename Connections>
    size_t send_all_in(Connections& collection, const segment_t& seg)
    {
        for (typename Connections::iterator it = collection.begin(), end = collection.end();
             it != end;
             ++it)
        {
            it->second->send(seg);
        }
        return collection.size();
    }

    outgoing_pool_ptr find_or_create_nolock(const host_info& address)
    {
        typename outgoing_map_t::iterator it_pool = outgoing_.find(address);
        if (it_pool != outgoing_.end())
        {
            return it_pool->second;
        }
        else
        {
            outgoing_pool_ptr new_pool =
                make_shared<outgoing_pool_t>(address, session_factory_, pool_settings_, stats_);
            new_pool->logger(this->logger());
            outgoing_.insert(it_pool, std::make_pair(address, new_pool));
            set_hooks(*new_pool);
            return new_pool;
        }
    }

    incoming_pool_ptr find_or_create_incoming_nolock(const host_info& address)
    {
        typename incoming_map_t::iterator it_pool = incoming_.find(address);
        if (it_pool != incoming_.end())
        {
            return it_pool->second;
        }
        else
        {
            incoming_pool_ptr new_pool = make_shared<incoming_pool_t>(address, stats_);
            new_pool->logger(this->logger());
            incoming_.insert(it_pool, std::make_pair(address, new_pool));
            set_hooks(*new_pool);
            return new_pool;
        }
    }

    void set_hooks(outgoing_pool_t& pool)
    {
        pool.set_hooks(
            boost::bind(&pool_manager::on_outgoing_connection_open, this->shared_from_this(), _1),
            boost::bind(&pool_manager::on_outgoing_connection_lost, this->shared_from_this(), _1),
            boost::bind(&pool_manager::on_message, this->shared_from_this(), _1, _2, _3));
    }

    void set_hooks(incoming_pool_t& pool)
    {
        pool.set_hooks(
            boost::bind(&pool_manager::on_incoming_connection_open, this->shared_from_this(), _1),
            boost::bind(&pool_manager::on_incoming_connection_lost, this->shared_from_this(), _1),
            boost::bind(&pool_manager::on_message, this->shared_from_this(), _1, _2, _3));
    }

    void send_resolved(const host_info& address, segment_t seg)
    {
        outgoing_pool_ptr pool = get_pool(address);
        if (pool)
        {
            pool->send(seg);
        }
        else
        {
            incoming_pool_ptr in_pool = get_incoming_pool(address);
            if (in_pool)
            {
                in_pool->send(seg);
            }
            else
            {
                throw std::logic_error("pool not opened to " + address.to_string());
            }
        }
    }

    outgoing_pool_ptr get_pool(const host_info& address)
    {
        lock_t lock(mux_);
        typename outgoing_map_t::iterator it_pool = outgoing_.find(address);
        return it_pool != outgoing_.end() ? it_pool->second : outgoing_pool_ptr();
    }

    incoming_pool_ptr get_incoming_pool(const host_info& address)
    {
        lock_t lock(mux_);
        typename incoming_map_t::iterator it_pool = incoming_.find(address);
        return it_pool != incoming_.end() ? it_pool->second : incoming_pool_ptr();
    }

    void on_resolve_back_incoming(const host_info& resolved_address, messenger_session_ptr session)
    {
        lock_t lock(mux_);
        incoming_pool_ptr pool = find_or_create_incoming_nolock(resolved_address);
        lock.unlock();
        pool->add_incoming_session(session);
    }

    void on_outgoing_connection_open(const host_info& info)
    {
        on_connection_open(info, outgoing_, pool_OUTGOING);
    }
    void on_incoming_connection_open(const host_info& info)
    {
        on_connection_open(info, incoming_, pool_INCOMING);
    }
    void on_outgoing_connection_lost(const host_info& info)
    {
        on_connection_lost(info, outgoing_, pool_OUTGOING);
    }
    void on_incoming_connection_lost(const host_info& info)
    {
        on_connection_lost(info, incoming_, pool_INCOMING);
    }

    template <typename Map>
    void on_connection_open(const host_info& info, Map& map, pool_type_t pool_type)
    {
        lock_t lock(mux_);
        typename Map::iterator it = map.find(info);
        if (it != map.end())
        {
            if (lock) lock.unlock();
            YLOG_L(info) << "pool " << info.to_string() << " connection opened";
            event_notification notification = { event_CONNECTED, pool_type };
            events_notifier_->notify(info.to_string(), notification);
        }
        else
        {
            YLOG_L(warning) << "on_connection_open " << info.to_string() << ": no such pool";
        }
    }

    template <typename Map>
    void on_connection_lost(const host_info& info, Map& map, pool_type_t pool_type)
    {
        string info_string = info.to_string();
        typename Map::mapped_type pool; // if any will be deleted after
        lock_t lock(mux_);
        typename Map::iterator it_pool = map.find(info);
        if (it_pool != map.end())
        {
            pool = it_pool->second;
            if (pool->is_empty())
            {
                map.erase(it_pool);
                lock.unlock();
                YLOG_L(info) << "pool " << info_string << " lost";
                pool->detach();
            }
            else
            {
                lock.unlock();
                YLOG_L(info) << "pool " << info_string << " connection lost";
            }
            event_notification notification = { event_POOL_LOST, pool_type };
            events_notifier_->notify(info_string, notification);
        }
        else
        {
            YLOG_L(debug) << "on_connection_lost " << info_string << ": no such pool";
        }
    }

    void on_message(const host_info& info, message_type type, const shared_buffers& buffers)
    {
        notifier_->notify(info.to_string(), type, buffers);
    }

    template <class Function>
    boost::function<void(void)> delayed_retry(Function&& func)
    {
        if (!io_) return [] {};

        auto ptr = this->shared_from_this();
        return [ptr, func, this] {
            if (!is_closed())
                make_delayed_call(pool_settings_.reconnect_delay, *io_, [ptr, func] { func(); });
        };
    }

    const std::shared_ptr<pool_resolver_t> pool_resolver_;
    const session_factory_ptr session_factory_;
    const shared_ptr<messages_notifier> notifier_;
    const shared_ptr<events_notifier> events_notifier_;

    pool_settings pool_settings_;
    outgoing_map_t outgoing_;
    incoming_map_t incoming_;
    mutex_t mux_;
    bool closed_;
    address_cache_t address_cache_;
    stats_ptr stats_;
    boost::asio::io_service* io_;
};

}
