#pragma once

#include <ymod_messenger/host_info.h>
#include <boost/optional.hpp>
#include <yplatform/coroutine.h>
#include <yplatform/yield.h>
#include <type_traits>
#include <stdexcept>

namespace ymod_messenger {

template <typename Resolver, typename ResolveOrder>
class pool_resolver : public yplatform::log::contains_logger
{
    template <typename Hook, typename ErrorHook>
    class resolve_coro;

public:
    using resolver_t = Resolver;
    using resolve_order_t = ResolveOrder;

    static_assert(
        std::is_enum<resolve_order_t>::value,
        "expected ResolveOrder to be an enumeration type");

    pool_resolver(
        boost::asio::io_service& io,
        resolver_t incoming,
        resolver_t outgoing,
        resolve_order_t outgoing_order,
        time_traits::duration timeout)
        : io_(io)
        , incoming_(std::move(incoming))
        , outgoing_(std::move(outgoing))
        , outgoing_order_(outgoing_order)
        , timeout_(timeout)
    {
    }

    void cancel()
    {
        incoming_.cancel();
        outgoing_.cancel();
    }

    template <typename Hook, typename ErrorHook>
    void resolve_incoming(const host_info& address, Hook&& hook, ErrorHook&& errhook)
    {
        using coro_t = resolve_coro<Hook, ErrorHook>;
        yplatform::spawn(std::make_shared<coro_t>(
            logger(),
            incoming_,
            std::forward<Hook>(hook),
            std::forward<ErrorHook>(errhook),
            address,
            steps(),
            timeout_,
            io_));
    }

    template <typename Hook, typename ErrorHook>
    void resolve_outgoing(const host_info& address, Hook&& hook, ErrorHook&& errhook)
    {
        using coro_t = resolve_coro<Hook, ErrorHook>;
        yplatform::spawn(std::make_shared<coro_t>(
            logger(),
            outgoing_,
            std::forward<Hook>(hook),
            std::forward<ErrorHook>(errhook),
            address,
            outgoing_steps(outgoing_order_),
            timeout_,
            io_));
    }

private:
    boost::asio::io_service& io_;
    resolver_t incoming_;
    resolver_t outgoing_;
    const resolve_order_t outgoing_order_;
    const time_traits::duration timeout_;

    enum class step
    {
        ipv6,
        ipv4
    };

    using steps = std::pair<boost::optional<step>, boost::optional<step>>;

    steps outgoing_steps(resolve_order_t order)
    {
        switch (order)
        {
        case resolve_order_t::ipv6_ipv4:
            return steps{ step::ipv6, step::ipv4 };
        case resolve_order_t::ipv4_ipv6:
            return steps{ step::ipv4, step::ipv6 };
        case resolve_order_t::ipv6:
            return steps{ step::ipv6, {} };
        case resolve_order_t::ipv4:
            return steps{ step::ipv4, {} };
        }

        using order_underlying_type = typename std::underlying_type<resolve_order_t>::type;
        throw std::invalid_argument(
            "Unknown resolve_order_t value " +
            std::to_string(static_cast<order_underlying_type>(order)));
    }

    template <typename Hook, typename ErrorHook>
    class resolve_coro : public yplatform::log::contains_logger
    {
        using yield_context = yplatform::yield_context<resolve_coro>;

    public:
        resolve_coro(
            const yplatform::log::source& logger,
            resolver_t& resolver,
            Hook&& hook,
            ErrorHook&& errhook,
            const host_info& request_host,
            const steps& forward_steps,
            const time_traits::duration& timeout,
            boost::asio::io_service& io)
            : yplatform::log::contains_logger(logger)
            , resolver_(resolver)
            , hook_(std::forward<Hook>(hook))
            , errhook_(std::forward<ErrorHook>(errhook))
            , request_host_(request_host)
            , forward_steps_(forward_steps)
            , timeout_(timeout)
            , strand_(io)
            , timer_(io)
        {
        }

        // Entry point. Should not be executed more than once.
        void operator()(yield_context ctx)
        {
            strand_.dispatch([ctx]() mutable { ctx(""); });
        }

        // Executed in strand.
        void operator()(yield_context ctx, const std::string& resolve_result)
        {
            if (result_reported_)
            {
                return;
            }

            try
            {
                reenter(ctx)
                {
                    start_timer(ctx);

                    if (!forward_steps_.first)
                    {
                        yield resolve_ptr(ctx);
                    }
                    else
                    {
                        yield resolve_forward(*forward_steps_.first, ctx);
                        if (resolve_result.empty() && forward_steps_.second)
                        {
                            yield resolve_forward(*forward_steps_.second, ctx);
                        }

                        if (!resolve_result.empty())
                        {
                            request_host_.addr = resolve_result;
                            yield resolve_ptr(ctx);
                        }
                    }

                    cancel_timer();
                    report_result(resolve_result);
                }
            }
            catch (const std::exception& e)
            {
                YLOG_L(error) << "pool_resolver exception: " << e.what();
                report_result("");
            }
            catch (...)
            {
                YLOG_L(error) << "pool_resolver unknown exception";
                report_result("");
            }
        }

        // Resolve handler. Executed in strand.
        template <typename ResolveIterator>
        void operator()(yield_context ctx, const boost::system::error_code& ec, ResolveIterator it)
        {
            if (ec || it == ResolveIterator())
            {
                std::string message = ec ? ec.message() : "no records";
                YLOG_L(error) << "failed to resolve address " << request_host_.to_string()
                              << " error: " << message;
                (*this)(ctx, "");
            }
            else
            {
                YLOG_L(debug) << request_host_.to_string() << " resolved to " << *it;
                (*this)(ctx, *it);
            }
        }

        // Timer handler. Executed in strand.
        void operator()(yield_context, const boost::system::error_code& ec)
        {
            if (!ec)
            {
                YLOG_L(error) << "failed to resolve address " << request_host_.to_string()
                              << " timed out";
                report_result("");
            }
        }

    private:
        void start_timer(yield_context ctx)
        {
            timer_.expires_from_now(timeout_);
            timer_.async_wait(strand_.wrap(ctx));
        }

        void resolve_forward(step s, yield_context ctx)
        {
            switch (s)
            {
            case step::ipv6:
                resolve_ipv6(ctx);
                return;
            case step::ipv4:
                resolve_ipv4(ctx);
                return;
            }

            throw std::invalid_argument(
                "Unknown step value " + std::to_string(static_cast<int>(s)));
        }

        void cancel_timer()
        {
            timer_.cancel();
        }

        void report_result(const std::string& result)
        {
            result_reported_ = true;
            try
            {
                if (!result.empty())
                {
                    hook_(host_info(result, request_host_.port));
                }
                else
                {
                    errhook_();
                }
            }
            catch (const std::exception& e)
            {
                YLOG_L(error) << "pool_resolver callback exception: " << e.what();
            }
            catch (...)
            {
                YLOG_L(error) << "pool_resolver callback unknown exception";
            }
        }

        void resolve_ipv6(yield_context ctx)
        {
            resolver_.async_resolve_aaaa(request_host_.addr, strand_.wrap(ctx));
        }

        void resolve_ipv4(yield_context ctx)
        {
            resolver_.async_resolve_a(request_host_.addr, strand_.wrap(ctx));
        }

        void resolve_ptr(yield_context ctx)
        {
            resolver_.async_resolve_ptr(request_host_.addr, strand_.wrap(ctx));
        }

        resolver_t& resolver_;
        Hook hook_;
        ErrorHook errhook_;
        host_info request_host_;
        steps forward_steps_;
        time_traits::duration timeout_;
        bool result_reported_{ false };
        boost::asio::io_service::strand strand_;
        time_traits::timer timer_;
    };
};

}

#include <yplatform/unyield.h>
