#pragma once

#include <boost/enable_shared_from_this.hpp>
#include <boost/system/error_code.hpp>

#include <yplatform/net/buffer_sequence.h>
#include <yplatform/task_context.h>
#include "../settings.h"
#include "stats.h"

namespace ymod_messenger {

enum err_code
{
    err_code_no_error = 0,        /// no error
    err_code_connection_lost = 1, /// one connection broken
    err_code_pool_lost = 2,       /// connection pool to host broken
};

/**
 * General session is used in pools
 */
class messenger_session_interface
{
public:
    typedef boost::function<
        void(shared_ptr<messenger_session_interface>, message_type, const shared_buffers&)>
        message_hook_t;
    typedef boost::function<void(shared_ptr<messenger_session_interface>, const error_code&)>
        error_hook_t;

    virtual bool is_open() const = 0;
    virtual void start_read() = 0;
    virtual void send(segment_t seg) = 0;
    virtual void async_close() = 0;
    virtual size_t send_queue_size() = 0;

    virtual void set_message_hook(const message_hook_t& hook) = 0;
    virtual void set_error_hook(const error_hook_t& hook) = 0;

    virtual session_stats_ptr get_stats() = 0;
    virtual string get_description() const = 0;

    virtual ~messenger_session_interface(){};
};

typedef shared_ptr<messenger_session_interface> messenger_session_ptr;

class messenger_session : public messenger_session_interface
{
public:
    /// TODO don't protect hooks
    virtual void set_message_hook(const message_hook_t& hook)
    {
        lock_t lock = get_lock();
        message_hook_ = hook;
    }

    virtual void set_error_hook(const error_hook_t& hook)
    {
        lock_t lock = get_lock();
        error_hook_ = hook;
    }

protected:
    typedef boost::mutex mutex_t;
    typedef boost::unique_lock<mutex_t> lock_t;

    inline lock_t get_lock()
    {
        return lock_t(mux_);
    }

    message_hook_t message_hook_;
    error_hook_t error_hook_;

private:
    mutex_t mux_;
};

}
