#pragma once

#include "stream_parser.h"
#include <boost/enable_shared_from_this.hpp>
#include <boost/bind/protect.hpp>

#include <yplatform/net/buffer_sequence.h>
#include <yplatform/net/session_strand.h>
#include <yplatform/task_context.h>

#include "../settings.h"
#include "messenger_session.h"
#include "segments_queue.h"

namespace ymod_messenger {

template <typename Parent, typename Settings>
class session_base
    : public Parent
    , public messenger_session
    , public boost::enable_shared_from_this<session_base<Parent, Settings>>
{
    typedef Parent base_t;

public:
    typedef Settings settings_t;

    session_base(yplatform::net::io_data* io_data, const settings_t& st)
        : base_t(io_data, st)
        , closed_(false)
        , write_active_(false)
        // TODO specify chunk sizes in config (but compare with defined in msgpack)
        , stats_(new session_stats)
        , stream_parser_()
    {
        std::stringstream stream;
        stream << "ymod_messenger_session " << base_t::id();
        base_t::set_log_prefix(stream.str());
    }

    bool is_closed()
    {
        lock_t lock = get_lock();
        return closed_;
    }

    void start_read() override
    {
        if (is_closed()) return;
        strand().post(boost::bind(&session_base::begin_read, this->shared_from_this()));
    }

    void send(segment_t seg) override
    {
        strand().post(
            std::bind(&session_base::real_send, this->shared_from_this(), std::move(seg)));
    }

    size_t send_queue_size() override
    {
        if (is_closed()) return 0;
        return writeq_.size();
    }

    void async_close() override
    {
        if (is_closed()) return;
        this->base_t::async_close(boost::bind(&session_base::do_nothing, this->shared_from_this()));
    }

    bool is_open() const override
    {
        return this->base_t::raw_socket().is_open();
    }

    session_stats_ptr get_stats() override
    {
        return stats_;
    }

    string get_description() const override
    {
        return base_t::get_log_prefix() + " local " + base_t::local_addr().to_string() + ":" +
            std::to_string(base_t::local_port()) + " remote " + base_t::remote_addr().to_string() +
            ":" + std::to_string(base_t::remote_port());
    }

protected:
    using base_t::strand;

    void begin_write();
    void after_write(const error_code& err, std::size_t bytes);
    void begin_read();
    void after_read(const error_code& err, std::size_t bytes);

    // in strand
    void real_send(segment_t& segment)
    {
        if (is_closed()) return;
        stats_->segments_sent++;
        stats_->send_queue = send_queue_size();
        writeq_.push(std::move(segment));
        begin_write();
    }

    // in strand
    void on_read_timeout()
    {
        NET_SESSION_LOG(info) << "read timeout";
        this->base_t::cancel_operations();
    }

    // in strand
    void on_write_timeout()
    {
        NET_SESSION_LOG(info) << "write timeout";
        this->base_t::cancel_operations();
    }

    // in strand
    void on_error(const error_code& err)
    {
        lock_t lock = get_lock();
        if (closed_) return;
        closed_ = true;
        error_hook_t error_hook = error_hook_;
        lock.unlock();
        this->do_shutdown(boost::bind(&session_base::do_nothing, this->shared_from_this()), false);

        if (error_hook) error_hook(this->shared_from_this(), err);
    }

    // in strand
    void on_message(message_type type, shared_buffers seq)
    {
        lock_t lock = get_lock();
        message_hook_t message_hook = message_hook_;
        lock.unlock();
        stats_->messages_received++;
        if (message_hook)
        {
            message_hook(this->shared_from_this(), type, std::move(seq));
        }
    }

    // TODO remove
    void do_nothing()
    {
    }

    bool closed_;

    bool write_active_;
    segments_queue writeq_;
    session_stats_ptr stats_;
    stream_parser stream_parser_;
};

// in strand
template <typename Parent, typename Settings>
void session_base<Parent, Settings>::begin_write()
{
    if (is_closed()) return;
    if (write_active_) return;
    try
    {
        if (!writeq_.flush(/*write chunk size*/)) return;
        write_active_ = true;
        base_t::async_write(
            writeq_.send_queue(),
            boost::bind(&session_base::after_write, this->shared_from_this(), _1, _2),
            boost::bind(&session_base::on_write_timeout, this->shared_from_this()));
    }
    catch (const std::exception& e)
    {
        NET_SESSION_LOG(error) << "begin_write error: " << e.what();
        // TODO on_error?
    }
}

// in strand
template <typename Parent, typename Settings>
void session_base<Parent, Settings>::after_write(const error_code& err, std::size_t bytes)
{
    stats_->bytes_sent += bytes;
    if (err)
    {
        on_error(err);
        return;
    }
    writeq_.consume(bytes);
    stats_->send_queue = send_queue_size();
    write_active_ = false;
    begin_write();
}

// in strand
template <typename Parent, typename Settings>
void session_base<Parent, Settings>::begin_read()
{
    if (is_closed()) return;
    try
    {
        stream_parser_.reserve_buffer();
    }
    catch (const std::exception& e)
    {
        NET_SESSION_LOG(error) << "buffer reserve error: " << e.what();
        on_error(boost::system::errc::make_error_code(boost::system::errc::no_buffer_space));
        return;
    }
    this->base_t::async_read(
        boost::bind(&session_base::after_read, this->shared_from_this(), _1, _2),
        boost::bind(&session_base::on_read_timeout, this->shared_from_this()),
        boost::asio::buffer(stream_parser_.buffer(), stream_parser_.buffer_free_size()));
}

// in strand
template <typename Parent, typename Settings>
void session_base<Parent, Settings>::after_read(const error_code& err, std::size_t bytes)
{
    stats_->bytes_received += bytes;
    if (err)
    {
        on_error(err);
        return;
    }

    stream_parser_.consume_buffer(bytes);
    message_type type;
    shared_buffers seq;
    try
    {
        while (stream_parser_.next(type, seq))
        {
            on_message(type, std::move(seq));
            type = 0;
            seq = shared_buffers();
        }
    }
    catch (const std::exception& e)
    {
        NET_SESSION_LOG(error) << "data stream error: " << e.what();
        on_error(boost::system::errc::make_error_code(boost::system::errc::protocol_error));
        return;
    }

    begin_read();
}

}
