#include <ymod_messenger/types.h>
#include <ymod_messenger/detail/header.h>
#include <boost/asio/buffer.hpp>

namespace ymod_messenger {
namespace detail {

struct unpack_context
{
    message_type type = 0;
    uint32_t buffer_size[MAX_SEND_SEQ_LEN];
    shared_buffer buffers[MAX_SEND_SEQ_LEN];
    bool header_parsed = false;
    unsigned seq_len = 0U;
    unsigned buffers_parsed = 0U;
};

template <typename T>
inline T unpack_numeric(const char* data, size_t& offset)
{
    T encoded;
    char* pencoded = static_cast<char*>(static_cast<void*>(&encoded));
    memcpy(pencoded, data, sizeof(T));
    offset += sizeof(T);
    return yplatform::net::network_2_host(encoded);
}

inline size_t unpack_header(const char* data, size_t size, unpack_context& ctx)
{
    if (size < header_size(1)) return 0U;
    size_t pos = 0U;
    ctx.type = unpack_numeric<message_type>(data + pos, pos);
    auto flags = unpack_numeric<flags_t>(data + pos, pos);
    ctx.seq_len = flags + 1;
    if (ctx.seq_len == 0) throw std::runtime_error("no fragments");
    if (ctx.seq_len > MAX_SEND_SEQ_LEN) throw std::runtime_error("too many fragments");
    if (size < header_size(ctx.seq_len)) return 0U;
    for (unsigned i = 0; i < ctx.seq_len; ++i)
    {
        ctx.buffer_size[i] = unpack_numeric<uint32_t>(data + pos, pos);
        if (ctx.buffer_size[i] > MAX_FRAGMENT_SIZE)
            throw std::runtime_error("fragment size exceeded");
    }
    ctx.header_parsed = true;
    return pos;
}

}

class stream_parser
{
public:
    stream_parser(size_t initial_buffer_size = 32 * 1024) : buffer_granularity_(initial_buffer_size)
    {
        size_ = buffer_granularity_;
        buffer_ = std::shared_ptr<char>(new char[size_], std::default_delete<char[]>());
        parse_begin_ = 0;
        parse_end_ = 0;
        ctx_ = detail::unpack_context();
    }

    char* buffer()
    {
        return buffer_.get() + parse_end_;
    }

    size_t buffer_free_size() const
    {
        return size_ - parse_end_;
    }

    void consume_buffer(size_t size)
    {
        assert(buffer_free_size() >= size);
        parse_end_ += size;
    }

    void reserve_buffer(size_t size = 16 * 1024)
    {
        // reuse current buffer if possible
        if (parse_begin_ > 0 && parse_begin_ == parse_end_ && buffer_.use_count() == 1)
        {
            parse_begin_ = 0;
            parse_end_ = 0;
        }
        if (buffer_free_size() < size)
        {
            do_realloc(size);
        }
    }

    bool next(message_type& type, shared_buffers& seq)
    {
        if (!ctx_.header_parsed)
        {
            if (!parse_header()) return false;
        }

        if (parse_binary())
        {
            type = ctx_.type;
            seq = shared_buffers(ctx_.buffers, ctx_.seq_len);
            ctx_ = detail::unpack_context();
            return true;
        }

        return false;
    }

    stream_parser(const stream_parser&) = delete;
    stream_parser& operator=(const stream_parser&) = delete;

private:
    void do_realloc(size_t size)
    {
        // TODO can do realloc if parse_begin_ == 0

        // allocate doubled size if possible
        size_t next_size = buffer_granularity_;
        size_t busy_block_size = parse_end_ - parse_begin_;
        while (next_size < size + busy_block_size)
        {
            size_t doubled_next_size = next_size * 2;
            if (doubled_next_size <= next_size)
            {
                next_size = size + busy_block_size;
                break;
            }
            next_size = doubled_next_size;
        }

        std::shared_ptr<char> new_buffer(new char[next_size], std::default_delete<char[]>());
        memcpy(new_buffer.get(), buffer_.get() + parse_begin_, busy_block_size);
        buffer_.swap(new_buffer);
        size_ = next_size;
        parse_end_ -= parse_begin_;
        parse_begin_ = 0;
    }

    bool parse_header()
    {
        size_t offset =
            detail::unpack_header(buffer_.get() + parse_begin_, parse_end_ - parse_begin_, ctx_);
        if (offset) parse_begin_ += offset;
        return ctx_.header_parsed;
    }

    bool parse_binary()
    {
        assert(ctx_.header_parsed);
        while (ctx_.buffers_parsed < ctx_.seq_len)
        {
            auto index = ctx_.buffers_parsed;
            auto buffer_size = ctx_.buffer_size[index];
            if (parse_end_ - parse_begin_ < buffer_size) break;
            ctx_.buffers[index] = shared_buffer(buffer_, buffer_.get() + parse_begin_, buffer_size);
            parse_begin_ += buffer_size;
            ctx_.buffers_parsed++;
        }
        return ctx_.buffers_parsed == ctx_.seq_len;
    }

    size_t buffer_granularity_;
    detail::unpack_context ctx_;
    std::shared_ptr<char> buffer_;
    size_t size_;
    size_t parse_begin_;
    size_t parse_end_;
};

}
