#pragma once

#include <boost/enable_shared_from_this.hpp>
#include <ymod_messenger/types.h>
#include <ymod_messenger/host_info.h>
#include "session/client_session.h"
#include "session/messenger_session.h"
#include "timer.h"

namespace ymod_messenger {

class session_factory_interface
{
public:
    typedef shared_ptr<messenger_session_interface> session_ptr;
    typedef boost::function<void(session_ptr, const error_code&)> connect_hook_t;
    typedef boost::function<void()> timeout_hook_t;

    virtual void create_session(
        const host_info& address,
        const time_traits::duration& delay,
        const connect_hook_t& connect_hook,
        const timeout_hook_t& timeout_hook) = 0;

    virtual ~session_factory_interface()
    {
    }
};

typedef shared_ptr<session_factory_interface> session_factory_ptr;

template <typename ClientModule>
class session_factory
    : public session_factory_interface
    , public boost::enable_shared_from_this<session_factory<ClientModule>>
{
    typedef ClientModule client_module;

public:
    session_factory(shared_ptr<client_module> client) : client_(client)
    {
    }

    void create_session(
        const host_info& address,
        const time_traits::duration& delay,
        const connect_hook_t& connect_hook,
        const timeout_hook_t& timeout_hook)
    {
        if (delay != time_traits::duration::min())
        {
            make_delayed_call(
                delay,
                *client_->io(),
                boost::protect(boost::bind(
                    &session_factory::create_session_impl,
                    this->shared_from_this(),
                    address,
                    connect_hook,
                    timeout_hook)));
        }
        else
        {
            create_session_impl(address, connect_hook, timeout_hook);
        }
    }

private:
    void create_session_impl(
        const host_info& address,
        const connect_hook_t& connect_hook,
        const timeout_hook_t& timeout_hook)
    {
        shared_ptr<client_session> session = client_->create_session();
        session->connect(
            address.addr,
            static_cast<unsigned short>(address.port),
            boost::bind(connect_hook, session, _1),
            timeout_hook);
    }

    shared_ptr<client_module> client_;
};

}
