#pragma once

#include <ymod_messenger/types.h>
#include <yplatform/net/settings.h>

namespace ymod_messenger {

struct settings_base : yplatform::log::contains_logger
{

    settings_base()
        : read_chunk(512 * 1024), min_fragmentation(64 * 1024), max_fragmentation(1024 * 1024)
    {
    }

    void parse_ptree(const boost::property_tree::ptree& data)
    {
        read_chunk = data.get("read_chunk", read_chunk);
        min_fragmentation = data.get("min_fragmentation", min_fragmentation);
        max_fragmentation = data.get("max_fragmentation", max_fragmentation);
        YLOG_L(debug) << "read_chunk=" << read_chunk;
        YLOG_L(debug) << "min_fragmentation=" << min_fragmentation;
        YLOG_L(debug) << "max_fragmentation=" << max_fragmentation;
    }

    unsigned read_chunk;
    unsigned min_fragmentation;
    unsigned max_fragmentation;
};

struct client_settings
    : public settings_base
    , public yplatform::net::client_settings
{
    void parse_ptree(const boost::property_tree::ptree& data)
    {
        settings_base::parse_ptree(data);
        ::yplatform::net::client_settings::parse_ptree(data);
    }
};

class server_session;

struct server_settings
    : public settings_base
    , public yplatform::net::server_settings
{

    struct
    {
        boost::function<void(const shared_ptr<server_session>&)> on_server_connection;
    } deps_injection;

    void parse_ptree(const boost::property_tree::ptree& data)
    {
        settings_base::parse_ptree(data);
        ::yplatform::net::server_settings::parse_ptree(data);
        YLOG_L(debug) << "tcp_no_delay=" << tcp_no_delay;
    }
};

struct pool_settings : yplatform::log::contains_logger
{
    pool_settings() : min_size(1), max_size(1), reconnect_delay(time_traits::seconds(3))
    {
    }

    void parse_ptree(const boost::property_tree::ptree& data)
    {
        min_size = data.get<size_t>("pool_min_size", min_size);
        max_size = data.get<size_t>("pool_max_size", max_size);
        reconnect_delay =
            time_traits::milliseconds(data.get<unsigned>("pool_reconnect_delay", 3000U));

        min_size = (min_size > 1 ? min_size : 1);
        max_size = (max_size > min_size ? max_size : min_size);

        YLOG_L(debug) << "pool_settings min_size=" << min_size;
        YLOG_L(debug) << "pool_settings max_size=" << max_size;
        YLOG_L(debug)
            << "pool_settings reconnect_delay="
            << time_traits::duration_cast<time_traits::milliseconds>(reconnect_delay).count();
    }

    size_t min_size;
    size_t max_size;
    time_traits::duration reconnect_delay;
};

}
