#pragma once

#include <ymod_messenger/types.h>
#include <yplatform/net/handlers/timer_handler.h>

namespace ymod_messenger {

using yplatform::net::timer_ptr;
using yplatform::net::timer_t;

template <typename TimerHook>
class timer_handler
{
public:
    timer_handler(TimerHook hook) : hook_(hook)
    {
    }

    void operator()(const error_code& e)
    {
        if (e != boost::asio::error::operation_aborted)
        {
            try
            {
                hook_();
            }
            catch (std::exception& e)
            {
                YLOG_G(error) << "timer_handler exception: " << e.what();
            }
        }
    }

private:
    TimerHook hook_;
};

template <typename TimerHook>
class delayed_call_handler
{
public:
    delayed_call_handler(TimerHook hook, timer_ptr timer) : hook_(hook), timer_(timer)
    {
    }

    void operator()(const error_code& e)
    {
        if (e != boost::asio::error::operation_aborted)
        {
            try
            {
                hook_();
            }
            catch (std::exception& e)
            {
                YLOG_G(error) << "timer_handler exception: " << e.what();
            }
        }
    }

private:
    TimerHook hook_;
    timer_ptr timer_;
};

template <typename TimerHook>
inline timer_handler<TimerHook> make_timer_handler(TimerHook hook)
{
    return timer_handler<TimerHook>(hook);
}

template <typename TimerHook>
inline delayed_call_handler<TimerHook> make_delayed_call_handler(TimerHook hook, timer_ptr timer)
{
    return delayed_call_handler<TimerHook>(hook, timer);
}

template <typename TimerHook>
inline timer_ptr create_timer(
    const time_traits::duration& timeout,
    boost::asio::io_service& io_service,
    TimerHook hook)
{
    if (timeout == time_traits::duration::max()) return timer_ptr();

    timer_ptr timer(new timer_t(io_service));
    try
    {
        timer->expires_from_now(timeout);
    }
    catch (boost::system::system_error const& err)
    {
        YLOG_G(error) << "create_timer: timer.expires_from_now error: " << err.what();
    }
    timer->async_wait(make_timer_handler(hook));

    return timer;
}

template <typename TimerHook>
void make_delayed_call(
    const time_traits::duration& timeout,
    boost::asio::io_service& io_service,
    TimerHook hook)
{
    if (timeout == time_traits::duration::max())
        throw std::runtime_error("make_delayed_call bad timeout");

    timer_ptr timer(new timer_t(io_service));
    try
    {
        timer->expires_from_now(timeout);
    }
    catch (boost::system::system_error const& err)
    {
        YLOG_G(error) << "create_timer: timer.expires_from_now error: " << err.what();
    }
    timer->async_wait(make_delayed_call_handler(hook, timer));
}

inline void cancel_timer(timer_ptr timer)
{
    if (timer) timer->cancel();
}

}