#include "extract_metrics.h"
#include <boost/algorithm/string/case_conv.hpp>
#include <boost/algorithm/string/trim.hpp>
#include <boost/algorithm/string/predicate.hpp>
#include <boost/property_tree/json_parser.hpp>
#include <boost/asio/streambuf.hpp>
#include <queue>
#include <iostream>

namespace ymod_metricserver {

using std::string;
using yplatform::ptree;

namespace {

bool leaf_node(const ptree& ptree)
{
    return ptree.empty();
}

bool hgram_node(const string& name)
{
    return boost::ends_with(name, "_hgram");
}

bool is_numeric(const string& value)
{
    try
    {
        stold(value);
        return true;
    }
    catch (...)
    {
    }

    return false;
}

bool is_bool(const string& value)
{
    return value == "true" || value == "false";
}

string convert_bool_to_numeric_string(const string& value)
{
    if (!is_bool(value)) throw std::runtime_error("invalid value");
    return value == "true" ? "1" : "0";
}

bool is_raw_hgram(const ptree& ptree)
{
    if (leaf_node(ptree))
    {
        return false;
    }

    for (auto&& [name, child] : ptree)
    {
        if (!name.empty()) return false;
        if (!is_numeric(child.data())) return false;
    }

    return true;
}

bool is_bucket_hgram(const ptree& ptree)
{
    if (leaf_node(ptree))
    {
        return false;
    }

    for (auto&& [bucket_name, bucket] : ptree)
    {
        if (!bucket_name.empty()) return false;
        if (bucket.size() != 2) return false;
        if (!is_raw_hgram(bucket)) return false;
    }
    return true;
}

string get_numeric_or_empty(const ptree& ptree)
{
    if (is_numeric(ptree.data())) return ptree.data();
    if (is_bool(ptree.data())) return convert_bool_to_numeric_string(ptree.data());
    return {};
}

string format_hgram_from_json(const string& hgram)
{
    string formatted = hgram;
    const auto char_filter = [](char c) {
        return c == '\"' || c == '{' || c == '}' || c == ':' || c == '\n';
    };

    // delete excess symbols from json representation
    formatted.erase(
        std::remove_if(formatted.begin(), formatted.end(), char_filter), formatted.end());

    return "[" + formatted + "]";
}

string get_hgram_or_empty(const ptree& ptree)
{
    if (!is_raw_hgram(ptree) && !is_bucket_hgram(ptree))
    {
        return "";
    }
    boost::asio::streambuf write_buffer;
    std::ostream hgram_stream(&write_buffer);
    boost::property_tree::write_json(hgram_stream, ptree, false);
    std::ostringstream hgram;
    hgram << hgram_stream.rdbuf();
    return format_hgram_from_json(hgram.str());
}

string append_to_prefix(const string& prefix, const string& str)
{
    if (prefix.empty()) return str;
    if (str.empty()) return prefix;
    return prefix + "_" + str;
}

string format_name(const string& name)
{
    auto formatted = boost::algorithm::to_lower_copy(name);

    const auto is_acceptable_special_char = [](char c) { return c == '_' || c == '.' || c == '-'; };
    const auto char_filter = [is_acceptable_special_char](char c) {
        return !(isalnum(c) || is_acceptable_special_char(c));
    };

    std::replace_if(formatted.begin(), formatted.end(), char_filter, '_');
    boost::algorithm::trim_if(formatted, is_acceptable_special_char);

    bool is_previous_char_special = false;
    formatted.erase(
        std::remove_if(
            formatted.begin(),
            formatted.end(),
            [&is_previous_char_special, is_acceptable_special_char](char c) {
                auto is_previous_special = is_previous_char_special;
                is_previous_char_special = is_acceptable_special_char(c);
                return is_acceptable_special_char(c) && is_previous_special;
            }),
        formatted.end());
    return formatted;
}

void try_add_statistic_to_metrics(
    const string& statistic_name,
    const string& value,
    std::map<string, string>& metrics,
    std::set<string>& forbidden_names)
{
    auto metric_name = format_name(statistic_name);
    if (metric_name.empty() || value.empty()) return;

    if (forbidden_names.find(metric_name) != forbidden_names.end()) return;

    if (metrics.find(metric_name) != metrics.end())
    {
        forbidden_names.insert(metric_name);
        metrics.erase(metrics.find(metric_name));
        return;
    }

    metrics[metric_name] = value;
}

}

string extract_metrics(const ptree& ptree)
{
    std::map<string, string> metrics;
    std::set<string> forbidden_names;

    std::queue<std::pair<string, yplatform::ptree>> subptrees_with_prefix;
    subptrees_with_prefix.emplace("", ptree);

    while (subptrees_with_prefix.size())
    {
        auto& [prefix, subptree] = subptrees_with_prefix.front();
        for (auto&& [child_name, child] : subptree)
        {
            if (hgram_node(child_name))
            {
                auto hgram = get_hgram_or_empty(child);
                if (!hgram.empty())
                {
                    try_add_statistic_to_metrics(
                        append_to_prefix(prefix, child_name), hgram, metrics, forbidden_names);
                    continue;
                }
            }
            if (leaf_node(child))
            {
                try_add_statistic_to_metrics(
                    append_to_prefix(prefix, child_name),
                    get_numeric_or_empty(child),
                    metrics,
                    forbidden_names);
                continue;
            }
            subptrees_with_prefix.emplace(append_to_prefix(prefix, child_name), child);
        }
        subptrees_with_prefix.pop();
    }

    string result;
    for (auto& [metric_name, value] : metrics)
    {
        result += metric_name + " " + value + "\n";
    }
    return result;
}

}