#pragma once

#include <ymod_paxos/network/detail/find_index.h>
#include <ymod_paxos/network/netch.h>
#include <ymod_paxos/network/messages.h>
#include <ymod_paxos/types.h>
#include <tuple>

namespace ymod_paxos {

template <typename T>
inline void pack_numeric(T t, streambuf& buf)
{
    T encoded = yplatform::net::host_2_network(t);
    const char* pencoded = static_cast<char*>(static_cast<void*>(&encoded));
    buf.write(pencoded, sizeof(T));
}

template <typename T>
inline void unpack_numeric(const char* data, [[maybe_unused]] size_t size, T& t, size_t& offset)
{
    assert(offset + sizeof(t) <= size);
    T encoded;
    char* pencoded = static_cast<char*>(static_cast<void*>(&encoded));
    memcpy(pencoded, data + offset, sizeof(T));
    offset += sizeof(T);
    t = yplatform::net::network_2_host(encoded);
}

class paxos_network : public std::enable_shared_from_this<paxos_network>
{
public:
    typedef netch_address address_type;

private:
    std::shared_ptr<netch> impl;
    netch_message_code code_offset;

    // WARNING: append new messages to the list end to keep backward compartibility
    typedef std::tuple<
        multipaxos::prepare_message,
        multipaxos::promise_message,
        multipaxos::reject_message,
        multipaxos::accept_message,
        multipaxos::learn_message,
        multipaxos::sync_request_message,
        multipaxos::sync_response_message,
        multipaxos::master_announce_message,
        redirect_message,
        redirect_response_message>
        message_types_list;

    template <typename TypeList, typename T>
    struct type_index
    {
        static const int value = detail::find_index<
            TypeList,
            T,
            std::tuple_size<TypeList>::value - 1,
            std::is_same<
                typename std::tuple_element<std::tuple_size<TypeList>::value - 1, TypeList>::type,
                T>::value>::value;
    };

    template <typename Callback>
    struct learn_callback_wrapper
    {
        learn_callback_wrapper(Callback&& cb) : cb_(std::forward<Callback>(cb))
        {
        }

        void operator()(const address_type& address, const ymod_messenger::shared_buffers& buffers)
            const
        {
            ;
            ;
            ;
            assert(buffers.size() == 2);
            if (buffers.size() != 2) return;
            auto& fields_buffer = buffers[0];
            learn_message msg;
            size_t offset = 0;
            unpack_numeric(fields_buffer.data(), fields_buffer.size(), msg.ballot, offset);
            size_t acceptor_id_len = 0;
            unpack_numeric(fields_buffer.data(), fields_buffer.size(), acceptor_id_len, offset);
            msg.acceptor_id = std::string(fields_buffer.data() + offset, acceptor_id_len);
            offset += acceptor_id_len;
            unpack_numeric(
                fields_buffer.data(), fields_buffer.size(), msg.accepted_value.slot, offset);

            auto& value_buffer = buffers[1];
            msg.accepted_value.value = multipaxos::value_t(
                value_buffer.native(), value_buffer.data(), value_buffer.size());

            cb_(address, msg);
        }

        typename std::decay<Callback>::type cb_;
    };

    template <typename Callback>
    struct accept_callback_wrapper
    {
        accept_callback_wrapper(Callback&& cb) : cb_(std::forward<Callback>(cb))
        {
        }

        void operator()(const address_type& address, const ymod_messenger::shared_buffers& buffers)
            const
        {
            ;
            ;
            ;
            assert(buffers.size() == 2);
            if (buffers.size() != 2) return;
            auto& fields_buffer = buffers[0];
            accept_message msg;
            size_t offset = 0;
            unpack_numeric(fields_buffer.data(), fields_buffer.size(), msg.ballot, offset);
            unpack_numeric(fields_buffer.data(), fields_buffer.size(), msg.pvalue.slot, offset);
            auto& value_buffer = buffers[1];
            msg.pvalue.value = multipaxos::value_t(
                value_buffer.native(), value_buffer.data(), value_buffer.size());
            cb_(address, msg);
        }

        typename std::decay<Callback>::type cb_;
    };

    template <typename T, typename Callback>
    struct bind_switcher
    {
        static void bind(std::shared_ptr<netch> impl, Callback&& callback, netch_message_code code)
        {
            impl->bind_messages<T>(std::forward<Callback>(callback), code);
        }
    };

    template <typename Callback>
    struct bind_switcher<learn_message, Callback>
    {
        static void bind(std::shared_ptr<netch> impl, Callback&& callback, netch_message_code code)
        {
            impl->bind_messages(
                learn_callback_wrapper<Callback>(std::forward<Callback>(callback)), code);
        }
    };

    template <typename Callback>
    struct bind_switcher<accept_message, Callback>
    {
        static void bind(std::shared_ptr<netch> impl, Callback&& callback, netch_message_code code)
        {
            impl->bind_messages(
                accept_callback_wrapper<Callback>(std::forward<Callback>(callback)), code);
        }
    };

public:
    paxos_network(std::shared_ptr<netch> netch, netch_message_code message_base_type)
        : impl(netch), code_offset(message_base_type + ymod_messenger::message_type_USER)
    {
    }

    const address_type& my_address() const
    {
        return impl->my_address();
    }

    template <typename T, typename Callback>
    void bind(Callback&& callback)
    {
        int message_code = type_index<message_types_list, T>::value;
        auto wrapped_callback =
            [guard = yplatform::weak_from(this),
             callback = std::forward<Callback>(callback)](auto&& address, auto&& data) {
                if (guard.lock())
                {
                    callback(address, data);
                }
            };
        bind_switcher<T, decltype(wrapped_callback)>::bind(
            impl, std::move(wrapped_callback), code_offset + message_code);
    }

    template <typename M>
    void broadcast(const M& message);

    template <typename M>
    void send(const M& message, const address_type& addr)
    {
        int message_code = type_index<message_types_list, M>::value;
        impl->send(addr, message, code_offset + message_code);
    }
};

template <typename M>
inline void paxos_network::broadcast(const M& message)
{
    int message_code = type_index<message_types_list, M>::value;
    impl->send_all(message, code_offset + message_code);
}

template <>
inline void paxos_network::broadcast(const multipaxos::learn_message& message)
{
    streambuf buf;
    pack_numeric(message.ballot, buf);
    pack_numeric(message.acceptor_id.size(), buf);
    buf.write(message.acceptor_id.data(), message.acceptor_id.size());
    pack_numeric(message.accepted_value.slot, buf);
    auto packed = buf.detach();
    auto& value = message.accepted_value.value;
    ymod_messenger::shared_buffer buf1(packed.native(), packed.data(), packed.size());
    ymod_messenger::shared_buffer buf2(value.native(), value.data(), value.size());
    int message_code = type_index<message_types_list, multipaxos::learn_message>::value;
    impl->send_all(ymod_messenger::pool_INCOMING, buf1, buf2, code_offset + message_code);
}

template <>
inline void paxos_network::broadcast(const multipaxos::accept_message& message)
{
    streambuf buf;
    pack_numeric(message.ballot, buf);
    pack_numeric(message.pvalue.slot, buf);
    auto packed = buf.detach();
    auto& value = message.pvalue.value;
    ymod_messenger::shared_buffer buf1(packed.native(), packed.data(), packed.size());
    ymod_messenger::shared_buffer buf2(value.native(), value.data(), value.size());
    int message_code = type_index<message_types_list, multipaxos::accept_message>::value;
    impl->send_all(ymod_messenger::pool_OUTGOING, buf1, buf2, code_offset + message_code);
}

} // namespace ymod_paxos
