#pragma once

#include <ymod_paxos/error.h>
#include <ymod_paxos/packing.hpp>
#include <ymod_paxos/types.h>
#include <msgpack.hpp>

namespace ymod_paxos {

class operation
{
public:
    // Packs header only, payload will be serialized in external code.
    operation(uint32_t code, const std::string& uniq_id) : code_(code), uniq_id_(uniq_id)
    {
        streambuf buf;
        pack_buffer(*this, buf);
        serialized_offset_ = buf.size();
        serialized_ = buf.detach();
    }

    // Packs both header and payload.
    template <typename PayloadT>
    operation(uint32_t code, const std::string& uniq_id, PayloadT&& t)
        : code_(code), uniq_id_(uniq_id)
    {
        streambuf buf;
        pack_buffer(*this, buf);
        serialized_offset_ = buf.size();
        pack_buffer(t, buf);
        serialized_ = buf.detach();
    }

    uint32_t code() const
    {
        return code_;
    }

    const std::string& uniq_id() const
    {
        return uniq_id_;
    }

    static const multipaxos::value_t& serialize(const operation& op)
    {
        return op.serialized_;
    }

    static operation deserialize(const multipaxos::value_t& serialized)
    {
        operation result;
        msgpack::unpacked unpacked =
            msgpack::unpack(serialized.data(), serialized.size(), result.serialized_offset_);
        unpacked.get().convert(result);
        result.serialized_ = serialized;
        return result;
    }

    template <typename PayloadT>
    PayloadT convert_payload() const
    {
        size_t offset = serialized_offset_;
        msgpack::unpacked unpacked =
            msgpack::unpack(serialized_.data(), serialized_.size(), offset);
        PayloadT result;
        unpacked.get().convert(result);
        return result;
    }

    MSGPACK_DEFINE(code_, uniq_id_);

protected:
    operation() = default;

private:
    uint32_t code_ = 0U;             // high-level DB operation code
    string uniq_id_;                 // optional: context id which initiated the operation
    multipaxos::value_t serialized_; // contains 1) header (operation common fields) 2) payload
    size_t serialized_offset_ = 0;   // payload's offset in serialized_
};

}
