#pragma once

#include <multipaxos/types.h>
#include <msgpack.hpp>
#include <string>
#include <vector>

namespace ymod_paxos {

class streambuf
{
    using byte_t = multipaxos::byte_t;
    using shared_carray = multipaxos::shared_carray;

public:
    streambuf(size_t initsz = 8 * 1024)
        : impl_(multipaxos::allocate_shared_carray(initsz)), size_(0), capacity_(initsz)
    {
    }

    streambuf(const streambuf&) = delete;
    streambuf& operator=(const streambuf&) = delete;
    streambuf(streambuf&& other) = delete;
    streambuf& operator=(streambuf&& other) = delete;

    void write(const byte_t* buf, size_t len)
    {
        if (capacity() - size() < len)
        {
            reserve(len);
        }
        byte_t* data_begin = data() + size();
        memcpy(data_begin, buf, len);
        size_ += len;
    }

    byte_t* data()
    {
        return impl_.get();
    }

    const byte_t* data() const
    {
        return impl_.get();
    }

    size_t size() const
    {
        return size_;
    }

    size_t capacity() const
    {
        return capacity_;
    }

    multipaxos::value_t detach()
    {
        multipaxos::value_t result(impl_, size_);
        impl_.reset();
        size_ = 0;
        capacity_ = 0;
        return result;
    }

private:
    void reserve(size_t len)
    {
        size_t new_size = (capacity() > 0) ? capacity() * 2 : 8 * 1024;

        while (new_size < size() + len)
        {
            size_t tmp_new_size = new_size * 2;
            if (tmp_new_size <= new_size)
            {
                new_size = size() + len;
                break;
            }
            new_size = tmp_new_size;
        }

        auto tmp = multipaxos::allocate_shared_carray(new_size);
        memcpy(tmp.get(), impl_.get(), size_);
        impl_ = tmp;
        capacity_ = new_size;
    }

private:
    shared_carray impl_;
    size_t size_;
    size_t capacity_;
};

template <typename T>
std::string pack(const T& t, size_t size_hint = MSGPACK_SBUFFER_INIT_SIZE)
{
    msgpack::sbuffer buf(size_hint);
    msgpack::pack(buf, t);
    return std::string(buf.data(), buf.size());
}

template <typename T>
multipaxos::buffer_t pack_ex(const T& t, size_t size_hint = MSGPACK_SBUFFER_INIT_SIZE)
{
    streambuf buf(size_hint);
    msgpack::pack(buf, t);
    return buf.detach();
}

template <typename T, typename Buffer>
void pack_buffer(const T& t, Buffer& buf)
{
    msgpack::pack(buf, t);
}

template <typename T, typename V = std::string>
void unpack(const V& v, T& message)
{
    msgpack::unpacked unpacked_msg = msgpack::unpack(v.data(), v.size());
    unpacked_msg.get().convert(message);
}

template <typename T>
void unpack_buffe(const char* data, size_t size, T& message)
{
    msgpack::unpacked unpacked_msg = msgpack::unpack(data, size);
    unpacked_msg.get().convert(message);
}

}
