#include "node_impl.h"

#include "rlog/rlog.h"
#include "sync/lookup_client.h"
#include "sync/lookup_server.h"
#include "sync/catch_up_client.h"
#include "sync/catch_up_server.h"
#include <ymod_paxos/types.h>
#include <ymod_paxos/db.h>
#include <yplatform/find.h>
#include <set>
#include <vector>

using std::string;

namespace ymod_paxos {

const std::string DBSTATUS_OPEN = "open";
const std::string DBSTATUS_LOOKUP = "lookup";
const std::string DBSTATUS_SYNC = "sync";
const std::string DBSTATUS_CLOSED = "closed";

void node_impl::init_network(const yplatform::ptree& /*conf*/)
{
    netch_ = yplatform::find<netch, std::shared_ptr>("netch");
    sync_netch_ = yplatform::find<netch, std::shared_ptr>("detached_netch");
    lease_netch_ = yplatform::find<netch, std::shared_ptr>("lease_netch");
    YLOG_L(info) << "my network addr is " << netch_->my_address();
    YLOG_L(info) << "my sync network addr is " << sync_netch_->my_address();
    YLOG_L(info) << "my lease network addr is " << lease_netch_->my_address();
}

void node_impl::init(const yplatform::ptree& conf)
{
    auto reactor = yplatform::global_net_reactor;

    database_ =
        yplatform::find<abstract_database, std::shared_ptr>(conf.get("database", "database"));
    timers_ = std::make_shared<timers::queue>(reactor);
    strand_.reset(new boost::asio::io_service::strand(*reactor->io()));

    cluster_name_ = conf.get<string>("cluster_name");
    cluster_size_ = conf.get<unsigned>("cluster_size");

    init_network(conf);
    auto network = std::make_shared<paxos_network>(
        netch_, conf.get<netch_message_code>("paxos_message_code_base"));
    auto message_codes = sync_message_types(conf.get<netch_message_code>("sync_message_code_base"));
    auto timers = std::make_shared<timers::queue>(reactor);

    lease_node_ = yplatform::find<ymod_lease::node, std::shared_ptr>("lease_node");
    auto win_cb = boost::bind(&node_impl::handle_win_lease, this, _1, _2, _3);
    auto loose_cb = boost::bind(&node_impl::handle_lose_lease, this, _1);
    lease_node_->bind(cluster_name_, strand_->wrap(win_cb), strand_->wrap(loose_cb));
    rlog_settings rlog_st;
    rlog_st.load(conf.get_child("replicated_log"));
    rlog_ = std::make_shared<rlog_t>(network, reactor, rlog_st);
    rlog_->logger(logger());
    lookup_client_ = std::make_shared<lookup_client>(
        sync_netch_,
        message_codes,
        timers,
        conf.get<time_duration>("lookup_timeout"),
        conf.get<unsigned>("lookup_attempts"));
    lookup_client_->logger(logger());
    lookup_server_ = std::make_shared<lookup_server>(sync_netch_, message_codes);
    lookup_server_->logger(logger());
    catch_up_client_ = std::make_shared<catch_up_client>(
        sync_netch_, message_codes, timers, conf.get<time_duration>("sync_timeout"));
    catch_up_client_->logger(logger());
    catch_up_server_ = std::make_shared<catch_up_server>(sync_netch_, message_codes);
    catch_up_server_->logger(logger());

    service_timer_ = timers_->create_timer();
    service_timer_interval_ = seconds(conf.get("service_timer_interval_sec", 1));

    db_status_ = &DBSTATUS_LOOKUP;
}

void node_impl::start()
{
    stop_ = false;
    start_lookup();
    start_service_timer();
}

void node_impl::stop()
{
    stop_ = true;
    service_timer_->cancel();
    lease_node_->update_lease_time(cluster_name_, milliseconds(100));
    std::this_thread::sleep_for(milliseconds(200));
    close_replica();
    lookup_client_->stop();
    lookup_server_->stop();
    catch_up_client_->stop();
    catch_up_server_->stop();
}

void node_impl::perform(const task_context_ptr& ctx, operation op, std::shared_ptr<icaller> caller)
{
    auto uniq_id = op.uniq_id();
    if (master_available_)
    {
        rlog_->perform(ctx, std::move(op), caller);
        stats_.performed++;
    }
    else
    {
        caller->set_error(uniq_id, error(ErrorCode_CANCELED, "no master"));
        stats_.canceled_no_master++;
    }
}

bool node_impl::is_master()
{
    return is_master_;
}

void node_impl::reset_replica()
{
    if (stop_ || db_status_ != &DBSTATUS_OPEN) return;
    close_replica();
    start_lookup();
}

yplatform::ptree node_impl::get_stats() const
{
    yplatform::ptree dest;
    auto* db_status = db_status_.load();
    auto revision = database_->get_revision();
    dest.put("db_lookup", db_status == &DBSTATUS_LOOKUP);
    dest.put("db_sync", db_status == &DBSTATUS_SYNC);
    dest.put("db_open", db_status == &DBSTATUS_OPEN);
    dest.put("db_closed", db_status == &DBSTATUS_CLOSED);
    dest.put("inval_revision", revision == -1);
    dest.put("group_splits_cumulative", stats_.group_splits);
    dest.put("performed_cumulative", stats_.performed);
    dest.put("canceled_no_master_cumulative", stats_.canceled_no_master);
    if (revision != -1)
    {
        dest.put("revision_cumulative", revision);
    }
    dest.put("master_available", master_available_);
    dest.put("is_master", is_master_);
    return dest;
}

void node_impl::set_id(unsigned id)
{
    // TODO check not started?
    id_ = id;
    YLOG_L(info) << "node id is set to " << id;
}

void node_impl::start_lookup()
{
    db_status_ = &DBSTATUS_LOOKUP;
    lookup_server_->reset_source(database_->get_revision());
    catch_up_server_->stop();
    lookup_client_->lookup(
        cluster_size_ - 1,
        database_->get_revision(),
        [this](bool up_to_date) {
            YLOG_L(notice) << "lookup finished with up_to_date=" << up_to_date;
            if (!up_to_date)
            {
                start_catch_up();
            }
            else
            {
                open_replica();
            }
        },
        [this]() {
            YLOG_L(notice) << "lookup failed, try again";
            start_lookup();
        });
}

void node_impl::start_catch_up()
{
    db_status_ = &DBSTATUS_SYNC;
    lookup_server_->reset_source(database_->get_revision());
    catch_up_client_->catch_up(
        database_,
        [this]() {
            YLOG_L(notice) << "catch up finished";
            open_replica();
        },
        [this]() {
            YLOG_L(notice) << "catch up failed, starting lookup";
            start_lookup();
        });
}

void node_impl::open_replica()
{
    strand_->post([this]() {
        rlog_->open(id_, database_);
        start_master_election();
        db_status_ = &DBSTATUS_OPEN;
        lookup_server_->set_source(database_);
        catch_up_server_->start(database_);
        up_time_ = std::time(0);
    });
}

void node_impl::close_replica()
{
    strand_->post([this]() {
        rlog_->close();
        stop_master_election();
        db_status_ = &DBSTATUS_CLOSED;
        up_time_ = 0;
        lookup_server_->reset_source(database_->get_revision());
        catch_up_server_->stop();
    });
}

void node_impl::start_master_election()
{
    YLOG_L(notice) << "automatic master election " << (lease_node_ ? "on" : "off");
    lease_node_->start_acquire_lease(cluster_name_);
}

void node_impl::stop_master_election()
{
    lease_node_->stop_acquire_lease(cluster_name_);
}

bool node_impl::data_is_ok()
{
    return database_->is_ok();
}

void node_impl::handle_win_lease(
    const string& resource_name,
    const ymod_lease::node_id& node,
    const ymod_lease::ballot_t ballot)
{
    master_available_ = true;
    if (node != lease_node_->node_id())
    {
        YLOG_L(info) << "lease found "
                     << "('" << resource_name << "', b" << ballot << ", node=" << node << ")";
        set_not_master();
    }
    else
    {
        YLOG_L(info) << "lease win "
                     << "('" << resource_name << "', b" << ballot << ", node=" << node << ")";
        set_master();
    }
}

void node_impl::handle_lose_lease(const string& resource_name)
{
    YLOG_L(info) << "lease lost "
                 << "(" << resource_name << ")";
    master_available_ = false;
    stats_.group_splits++;
    set_not_master();
}

void node_impl::set_master()
{
    if (is_master_) return;
    rlog_->set_master();
    catch_up_server_->set_master();
    lookup_server_->set_master();
    is_master_ = true;
}

void node_impl::set_not_master()
{
    rlog_->drop_callers();
    if (!is_master_) return;
    rlog_->set_not_master();
    catch_up_server_->set_master(false);
    lookup_server_->set_master(false);
    is_master_ = false;
}

node::status node_impl::get_status()
{
    node::status result;
    result.status = *db_status_.load();
    result.revision = database_->get_revision();
    result.master_available = master_available_;
    result.is_master = is_master_;
    result.up_time = up_time_;
    return result;
}

void node_impl::start_service_timer()
{
    if (stop_) return;

    if (service_timer_)
    {
        service_timer_->async_wait(
            service_timer_interval_, boost::bind(&node_impl::handle_service_timer, this));
    }
}

void node_impl::handle_service_timer()
{
    if (stop_) return;
    if (db_status_.load() == &DBSTATUS_OPEN)
    {
        if (rlog_->is_lagging())
        {
            YLOG_L(error) << "reset lagging relica";
            reset_replica();
        }
    }
    start_service_timer();
}

}
