#pragma once

#include <ymod_paxos/types.h>
#include <sintimers/timer.h>
#include <sintimers/queue.h>

namespace ymod_paxos {

template <typename task_id_t = std::string>
class task_repeater
{
public:
    typedef boost::function<void(const task_id_t& task_id)> on_task_repeat_callback_t;

public:
    task_repeater(
        const std::shared_ptr<timers::queue>& timers_queue,
        const on_task_repeat_callback_t& callback)
        : timers_queue_(timers_queue), on_task_repeat_(callback)
    {
    }

    void push(const task_id_t& task_id, const time_duration& repeat_interval)
    {
        lock_t lock(mutex_);
        auto task_timer = task_timers_.find(task_id);
        if (task_timer == task_timers_.end())
        {
            timers::timer_ptr timer = timers_queue_->create_timer();
            timer->async_wait(
                repeat_interval, boost::bind(&task_repeater::on_timer, this, task_id));
            task_timers_.insert(task_timer, std::make_pair(task_id, timer));
            lock.unlock();
        }
    }

    void on_timer(const task_id_t& task_id)
    {
        on_task_repeat_(task_id);
        erase(task_id);
    }

    void clear()
    {
        lock_t lock(mutex_);
        timers_collection_t old_timers;
        task_timers_.swap(old_timers);
        lock.unlock();

        for (auto& pair : old_timers)
            pair.second->cancel();
    }

    bool has(const task_id_t& task_id)
    {
        lock_t lock(mutex_);
        return task_timers_.count(task_id);
    }

    size_t size()
    {
        lock_t lock(mutex_);
        return task_timers_.size();
    }

    void erase(const task_id_t& task_id)
    {
        lock_t lock(mutex_);
        auto timer = task_timers_.find(task_id);

        if (timer == task_timers_.end()) return;

        timer->second->cancel();
        task_timers_.erase(timer);
        lock.unlock();
    }

private:
    typedef std::map<task_id_t, timers::timer_ptr> timers_collection_t;

    timers_collection_t task_timers_;
    std::shared_ptr<timers::queue> timers_queue_;
    mutex_t mutex_;

    on_task_repeat_callback_t on_task_repeat_;
};
}
