#include "rlog.h"

#include <ymod_paxos/db.h>
#include <ymod_paxos/error.h>
#include <ymod_paxos/packing.hpp>
#include <ymod_paxos/types.h>

#include <yplatform/find.h> // find database
#include <sintimers/queue.h>
#include <boost/bind.hpp>
#include <msgpack.hpp>

namespace ymod_paxos {

namespace {
struct exec_wrapper
{
    exec_wrapper(
        operation&& op,
        std::shared_ptr<icaller> caller,
        iid_t iid,
        std::weak_ptr<abstract_database> database,
        std::atomic<bool>& exec_enabled)
        : op(std::move(op))
        , caller(caller)
        , iid(iid)
        , database(database)
        , exec_enabled(exec_enabled)
    {
    }

    // reactor will catch all exceptions
    void operator()()
    {
        auto pdatabase = database.lock();
        if (pdatabase && exec_enabled)
        {
            pdatabase->apply(iid, std::move(op), caller);
        }
        else if (caller)
        {
            caller->set_error(op.uniq_id(), error(ErrorCode_CANCELED, "operation canceled"));
        }
    }

    operation op;
    std::shared_ptr<icaller> caller;
    iid_t iid;
    std::weak_ptr<abstract_database> database;
    std::atomic<bool>& exec_enabled;
};

struct cancel_wrapper
{
    cancel_wrapper(const string& uniq_id, std::shared_ptr<icaller> caller)
        : uniq_id(uniq_id), caller(caller)
    {
    }

    // reactor will catch all exceptions
    void operator()()
    {
        caller->set_error(uniq_id, error(ErrorCode_CANCELED, "operation canceled"));
    }

    string uniq_id;
    std::shared_ptr<icaller> caller;
};
}

rlog_t::rlog_t(
    std::shared_ptr<paxos_network> network,
    yplatform::reactor_ptr reactor,
    const rlog_settings& settings)
{
    opened_ = false;
    is_master_ = false;
    lagging_ = false;

    network_ = network;
    reactor_ = reactor;
    settings_ = settings;

    timers_queue_ = std::make_shared<timers::queue>(reactor_);
    exec_strand_.reset(new boost::asio::io_service::strand(*reactor_->io()));

    stats_ = std::make_shared<rlog_stats>();

    network_->template bind<redirect_message>(boost::bind(&rlog_t::handle_redirect, this, _1, _2));
    network_->template bind<redirect_response_message>(
        boost::bind(&rlog_t::handle_redirect_response, this, _1, _2));

    redirects_ = std::make_shared<task_repeater<>>(
        timers_queue_, boost::bind(&rlog_t::handle_redirect_timeout, this, _1));

    if (settings_.agent_log_enabled)
    {
        auto custom_logger = yplatform::log::source(YGLOBAL_LOG_SERVICE, settings_.paxos_log_id);
        auto log_level =
            settings_.agent_debug_log_enabled ? ylogger::level::debug : ylogger::level::info;
        auto global_reactor = yplatform::global_net_reactor;
        replication_agent_.reset(new replication_agent_type(ylogger(custom_logger, log_level)));
    }
    else
    {
        replication_agent_.reset(new replication_agent_type(ylogger()));
    }
    replication_agent_->init_network(network_);
}

void rlog_t::open(int replica_id, std::shared_ptr<abstract_database> database)
{
    lock_t lock(mutex_);
    if (opened_)
    {
        throw std::runtime_error("rlog_t::open failed: is already opened");
    }
    auto revision = database->get_revision() + 1;
    database_ = database;
    opened_ = true;
    lagging_ = false;
    lock.unlock();

    try
    {
        auto deliver_f = boost::bind(&rlog_t::handle_replication_deliver, this, _1, _2);
        auto drop_f = boost::bind(&rlog_t::handle_replication_drop, this, _1);
        auto report_f = boost::bind(&rlog_t::handle_replication_report, this, _1);
        replication_agent_->init(
            settings_.algorithm,
            replica_id,
            settings_.total_log_size,
            revision,
            deliver_f,
            drop_f,
            report_f);
        replication_agent_->start();
    }
    catch (...)
    {
        lock.lock();
        database_.reset();
        opened_ = false;
        throw;
    }

    YLOG_L(info) << "replica opened: revision=" << revision;
}

void rlog_t::close()
{
    set_not_master();
    lock_t lock(mutex_);
    if (!opened_) return;

    opened_ = false;
    // exec wrappers in reactor refer db by weak_ptr, so db reset
    // prevents operations to be applied to db after replica close
    database_.reset();
    drop_callers(lock); // unlocks
    replication_agent_->stop();
}

void rlog_t::perform(const task_context_ptr& ctx, operation op, std::shared_ptr<icaller> caller)
{
    // caller's exceptions will be threw up to stack, it's ok

    lock_t lock(mutex_);
    if (!opened_)
    {
        lock.unlock();
        caller->set_error(op.uniq_id(), error(ErrorCode_DOWN, "server is closed"));
        return;
    }
    if (ctx->is_cancelled())
    {
        lock.unlock();
        caller->set_error(op.uniq_id(), error(ErrorCode_CANCELED, "operation canceled"));
        return;
    }
    if (callers_.count(op.uniq_id()))
    {
        lock.unlock();
        YLOG_L(info) << "attepmt to perform a duplicate uniq_id=" << op.uniq_id();
        caller->set_error(op.uniq_id(), error(ErrorCode_DUPLICATE, "duplicate"));
        return;
    }

    if (!database_->is_modifying(op))
    {
        if (is_master_)
        {
            lock.unlock();
            exec_dirty(std::move(op), caller);
            return;
        }
    }

    auto uniq_id = op.uniq_id();
    try
    {
        save_caller(op.uniq_id(), caller);
        if (is_master_)
        {
            lock.unlock();
            replication_propose(std::move(op));
        }
        else
        {
            auto timeout =
                std::min(ctx->deadline() - clock::now(), settings_.default_redirect_timeout);
            redirects_->push(uniq_id, timeout);
            lock.unlock();
            redirect(std::move(op));
        }
    }
    catch (std::exception& e)
    {
        YLOG_L(error) << "broadcast error: " << e.what();
        restore_saved_caller(uniq_id);
        caller->set_error(uniq_id, error(ErrorCode_CANT_BROADCAST, e.what()));
    }
}

void rlog_t::drop_callers(lock_t& lock)
{
    callers_map_type copy;
    copy.swap(callers_);
    redirects_->clear();
    lock.unlock();
    for (auto& pair : copy)
    {
        cancel(pair.first, pair.second);
    }
}

void rlog_t::drop_callers()
{
    lock_t lock(mutex_);
    drop_callers(lock);
}

void rlog_t::redirect(operation op)
{
    YLOG_L(debug) << "redirecting uniq_id=" << op.uniq_id();
    stats_->redirects_sent++;
    redirect_message msg;
    msg.op_uniq_id = op.uniq_id();
    msg.op = operation::serialize(op);
    network_->broadcast(msg);
}

void rlog_t::handle_redirect_timeout(const string& uniq_id)
{
    lock_t lock(mutex_);
    if (is_master_)
    {
        lock.unlock();
        YLOG_L(debug) << "became master, stop redirecting original_id=" << uniq_id;
        return;
    }
    auto found = callers_.find(uniq_id);
    if (found == callers_.end())
    {
        lock.unlock();
        YLOG_L(debug) << "uniq_id=" << uniq_id << " has gone away";
        return;
    }
    auto caller = found->second;
    callers_.erase(found);
    lock.unlock();
    YLOG_L(debug) << "uniq_id=" << uniq_id << " is timed out";
    cancel(uniq_id, caller);
}

void rlog_t::handle_replication_deliver(iid_t iid, multipaxos::value_t value)
{
    operation op = operation::deserialize(value);
    lock_t lock(mutex_);
    auto caller = restore_saved_caller(op.uniq_id());
    auto database = database_;
    lock.unlock();
    exec_strand_->post(exec_wrapper(std::move(op), caller, iid, database, opened_));
    stats_->executed++;
}

void rlog_t::handle_replication_drop(multipaxos::value_t value)
{
    operation op = operation::deserialize(value);
    lock_t lock(mutex_);
    auto caller = restore_saved_caller(op.uniq_id());
    if (caller)
    {
        lock.unlock();
        cancel(op.uniq_id(), caller);
    }
}

void rlog_t::handle_replication_report(multipaxos::report_code code)
{
    if (code == multipaxos::report_code::lagging)
    {
        lagging_ = true;
    }
}

void rlog_t::handle_redirect(const network_address& sender, const redirect_message& msg)
{
    lock_t lock(mutex_);
    if (!opened_ || !is_master_) return;
    stats_->redirects_received++;
    lock.unlock();
    auto op = operation::deserialize(msg.op);
    auto caller = std::make_shared<ymod_paxos::redirect_caller>(network_, sender);
    YLOG_L(debug) << "received redirected operation from=" << sender << " uniq_id=" << op.uniq_id();
    perform(boost::make_shared<yplatform::task_context>(), std::move(op), caller);
}

void rlog_t::handle_redirect_response(
    const network_address& sender,
    const redirect_response_message& msg)
{
    lock_t lock(mutex_);
    if (!opened_ || is_master_) return;
    YLOG_L(debug) << "received result for redirected operation uniq_id=" << msg.op_uniq_id
                  << " from=" << sender;
    auto caller = restore_saved_caller(msg.op_uniq_id);
    lock.unlock();
    if (caller)
    {
        for (const auto& attr : msg.attributes)
        {
            caller->set_attribute(attr.first, attr.second);
        }
        caller->set_result(msg.op_uniq_id, msg.data);
    }
}

void rlog_t::save_caller(const string& uniq_id, std::shared_ptr<icaller> caller)
{
    callers_[uniq_id] = caller;
}

std::shared_ptr<icaller> rlog_t::restore_saved_caller(const string& uniq_id)
{
    std::shared_ptr<icaller> result;
    auto found = callers_.find(uniq_id);
    if (found != callers_.end())
    {
        result = found->second;
        callers_.erase(found);
        if (!is_master_)
        {
            redirects_->erase(uniq_id);
        }
    }
    return result;
}

bool rlog_t::is_master()
{
    lock_t lock(mutex_);
    return is_master_;
}

bool rlog_t::is_opened()
{
    lock_t lock(mutex_);
    return opened_;
}

void rlog_t::set_master()
{
    lock_t lock(mutex_);
    if (!opened_) throw std::runtime_error("rlog_t::set_master failed: not opened yet");
    if (is_master_) throw std::runtime_error("rlog_t::set_master failed: is a master");
    is_master_ = true;
    replication_agent_->set_master();
    redirects_->clear();
    callers_map_type copy;
    copy.swap(callers_);
    lock.unlock();
    for (auto& pair : copy)
    {
        // TODO store redirected ops and re-perform as master
        cancel(pair.first, pair.second);
    }
}

void rlog_t::set_not_master()
{
    lock_t lock(mutex_);
    if (!opened_) return;
    if (!is_master_) return;
    is_master_ = false;
    replication_agent_->set_not_master();
    drop_callers(lock);
}

void rlog_t::exec_dirty(operation op, std::shared_ptr<icaller> caller)
{
    lock_t lock(mutex_);
    auto database = database_;
    lock.unlock();
    caller->set_dirty_call();
    exec_strand_->get_io_service().post(exec_wrapper(std::move(op), caller, -1, database, opened_));
    stats_->executed++;
}

void rlog_t::cancel(const string& uniq_id, std::shared_ptr<icaller> caller)
{
    exec_strand_->post(cancel_wrapper(uniq_id, caller));
    stats_->canceled++;
}

void rlog_t::replication_propose(operation op)
{
    auto value = operation::serialize(op);
    replication_agent_->submit(value);
}

std::shared_ptr<rlog_stats> rlog_t::stats() const
{
    lock_t lock(mutex_);
    if (opened_)
    {
        const auto& agent_stats = replication_agent_->get_stats();
        stats_->multipaxos_stats.submits = agent_stats.submits.load();
        stats_->multipaxos_stats.learns_received = agent_stats.learns_received.load();
        stats_->multipaxos_stats.learns_skipped_wrong_ballot =
            agent_stats.learns_skipped_wrong_ballot.load();
        stats_->multipaxos_stats.reset_prepare = agent_stats.reset_prepare.load();
    }
    lock.unlock();
    return stats_;
}

yplatform::ptree rlog_t::get_stats() const
{
    yplatform::ptree dest;
    stats_->to_ptree(dest);
    return dest;
}

}
