#pragma once

#include "lru.h"
#include "redirect_map.h"
#include "redirect_caller.h"
#include "settings.h"
#include "stats.h"
#include "ylogger.h"
#include <ymod_paxos/db.h>
#include <ymod_paxos/network.h>
#include <ymod_paxos/operation.h>
#include <ymod_paxos/types.h>
#include <multipaxos/ts_agent.h>
#include <sintimers/queue.h>

namespace ymod_paxos {

class rlog_t : public yplatform::log::contains_logger
{
    using replication_agent_type = multipaxos::ts_agent<paxos_network, ylogger>;
    using callers_map_type = std::map<string, std::shared_ptr<icaller>>;
    using network_address = paxos_network::address_type;

public:
    rlog_t(
        std::shared_ptr<paxos_network> network,
        yplatform::reactor_ptr reactor,
        const rlog_settings& settings);

    void open(int replica_id, std::shared_ptr<abstract_database> database);
    void close();

    void perform(const task_context_ptr& ctx, operation op, std::shared_ptr<icaller> caller);
    void set_master();
    void set_not_master();
    void drop_callers();

    bool is_master();
    bool is_opened();
    bool is_lagging()
    {
        return lagging_;
    }
    std::shared_ptr<rlog_stats> stats() const;
    yplatform::ptree get_stats() const;

private:
    void exec_dirty(operation op, std::shared_ptr<icaller> caller);
    void cancel(const string& uniq_id, std::shared_ptr<icaller> caller);

    void replication_propose(operation op);
    void handle_replication_deliver(iid_t iid, multipaxos::value_t value);
    void handle_replication_drop(multipaxos::value_t value);
    void handle_replication_report(multipaxos::report_code code);

    void redirect(operation op);
    void handle_redirect(const network_address& sender, const redirect_message& msg);
    void handle_redirect_response(
        const network_address& sender,
        const redirect_response_message& msg);
    void handle_redirect_timeout(const string& uniq_id);

    void save_caller(const string& uniq_id, std::shared_ptr<icaller> caller);
    std::shared_ptr<icaller> restore_saved_caller(const string& uniq_id);
    void drop_callers(lock_t& lock);

protected:
    mutable mutex_t mutex_;

    rlog_settings settings_;
    yplatform::reactor_ptr reactor_;
    callers_map_type callers_;
    std::shared_ptr<paxos_network> network_;
    std::shared_ptr<timers::queue> timers_queue_;
    std::shared_ptr<abstract_database> database_;
    std::shared_ptr<task_repeater<>> redirects_;
    std::unique_ptr<boost::asio::io_service::strand> exec_strand_;
    std::shared_ptr<replication_agent_type> replication_agent_;

    std::atomic<bool> is_master_;
    std::atomic<bool> opened_;
    std::atomic<bool> lagging_;

    std::shared_ptr<rlog_stats> stats_;
};

}
