#include "sync/catch_up_client.h"
#include <ymod_paxos/packing.hpp>

namespace ymod_paxos {

catch_up_client::catch_up_client(
    std::shared_ptr<netch> netch,
    const sync_message_types& sync_message_types,
    std::shared_ptr<timers::queue> timers,
    const time_duration& timeout,
    unsigned concurrency)
    : netch_(netch)
    , sync_messages_(sync_message_types)
    , timers_(timers)
    , timeout_(timeout)
    , concurrency_(concurrency)
{
    namespace ph = std::placeholders;
    stop_ = false;
    concurrency_ = std::max(std::min(1U, concurrency), 100U);
    netch_->bind_messages<abstract_database::serialized_data_type>(
        std::bind(&catch_up_client::handle_get_snapshot_answer, this, ph::_1, ph::_2),
        sync_messages_.catch_up_answer);
    netch_->bind_messages<abstract_database::serialized_data_type>(
        std::bind(&catch_up_client::handle_get_delta_answer, this, ph::_1, ph::_2),
        sync_messages_.catch_up_get_delta_answer);
    deadline_ = timers_->create_timer();
}

void catch_up_client::stop()
{
    stop_ = true;
    deadline_->cancel();
}

void catch_up_client::catch_up(
    std::shared_ptr<abstract_database> db,
    const CatchUpCallback& cb,
    const ErrorCallback& error_cb)
{
    lock_t lock(mutex_);
    if (session_)
    {
        throw std::runtime_error("catch up already running");
    }

    session_.reset(new catch_up_session);
    session_->db = db;
    session_->id = uuid_generator_();
    session_->success_cb = cb;
    session_->error_cb = error_cb;

    catch_up_message msg;
    msg.session_id = session_->id;

    lock.unlock();

    netch_->send_all(msg, sync_messages_.catch_up);

    deadline_->async_wait(timeout_, boost::bind(&catch_up_client::check_catch_up_deadline, this));
}

void catch_up_client::handle_get_snapshot_answer(
    const netch_address& addr,
    const abstract_database::serialized_data_type& snapshot)
{
    lock_t lock(mutex_);
    // check if session_ is not expired or snapshot is not already received
    if (!session_ || session_->snapshot_received())
    {
        return;
    }

    YLOG_L(info) << "catch_up_client: snapshot received from " << addr;

    session_->source = addr;
    session_->sync_manager = session_->db->start_sync(snapshot);

    deadline_->cancel();

    if (session_->sync_manager->finished())
    {
        YLOG_L(info) << "catch_up_client: empty diff, records are consistent";
        finish_catch_up(lock);
        return;
    }

    for (unsigned i = 0; i < concurrency_; ++i)
    {
        if (!lock) lock.lock();
        request_records(addr, lock);
    }
    deadline_->async_wait(timeout_, boost::bind(&catch_up_client::check_catch_up_deadline, this));
}

void catch_up_client::handle_get_delta_answer(
    const netch_address& addr,
    const abstract_database::serialized_data_type& msg)
{
    lock_t lock(mutex_);
    // check corresponding session_ not deleted
    if (!session_ || session_->source != addr)
    {
        return;
    }

    deadline_->cancel();
    session_->sync_manager->apply_delta(msg);
    request_records(addr, lock);
    deadline_->async_wait(timeout_, boost::bind(&catch_up_client::check_catch_up_deadline, this));
}

void catch_up_client::check_catch_up_deadline()
{
    if (stop_) return;

    lock_t lock(mutex_);
    if (session_)
    {
        YLOG_L(info) << "catch up session_ " << session_->id << " timeout";
        auto cb = session_->error_cb;
        session_.reset();
        lock.unlock();
        cb();
        return;
    }
}

size_t catch_up_client::remained()
{
    lock_t lock(mutex_);
    if (session_)
    {
        return session_->sync_manager->remained();
    }
    else
    {
        return 0;
    }
}

void catch_up_client::request_records(const netch_address& addr, lock_t& lock)
{
    assert(lock);
    if (!session_)
    {
        return;
    }

    if (session_->sync_manager->finished())
    {
        finish_catch_up(lock);
        return;
    }

    lock.unlock();

    auto request_msg = session_->sync_manager->make_delta_request();

    if (request_msg)
    {
        netch_->send(addr, *request_msg, sync_messages_.catch_up_get_delta);
    }
}

void catch_up_client::finish_catch_up(lock_t& lock)
{
    YLOG_L(debug) << "catch up is going to finish";
    assert(session_->finished());
    try
    {
        session_->db->finish_sync(session_->sync_manager);
    }
    catch (std::exception& e)
    {
        YLOG_L(error) << "finish_catch_up faield: " << e.what();
        // do not call error_cb, wait for timeout
        return;
    }
    auto cb = session_->success_cb;
    session_.reset();
    lock.unlock();
    cb();
}

}
