#include "sync/lookup_client.h"
#include <ymod_paxos/packing.hpp>

namespace ymod_paxos {

lookup_client::lookup_client(
    std::shared_ptr<netch> netch,
    const sync_message_types& sync_message_types,
    std::shared_ptr<timers::queue> timers,
    const time_duration& timeout,
    unsigned retry_count)
    : netch_(netch)
    , sync_messages_(sync_message_types)
    , timers_(timers)
    , timeout_(timeout)
    , retry_count_(retry_count)
{
    netch_->bind_messages<lookup_answer_message>(
        boost::bind(&lookup_client::on_message_lookup_answer, this, _1, _2),
        sync_messages_.lookup_answer);
    lookup_timer_ = timers_->create_timer();
    stop_ = false;
}

void lookup_client::lookup(
    size_t cluster_size,
    iid_t revision,
    const LookupCallback& cb,
    const ErrorCallback& error_cb)
{
    lock_t lock(mutex_);
    if (lookup_session_)
    {
        throw std::runtime_error("lookup already running");
    }
    lookup_session_.reset(new lookup_session);
    lookup_session_->cluster_size = cluster_size;
    lookup_session_->success_cb = cb;
    lookup_session_->error_cb = error_cb;
    lookup_session_->revision = revision;
    lookup_try(lock);
}

void lookup_client::stop()
{
    stop_ = true;
    lookup_timer_->cancel();
}

void lookup_client::on_message_lookup_answer(
    const netch_address& addr,
    const lookup_answer_message& msg)
{
    if (addr == netch_->my_address())
    {
        YLOG_L(notice) << "lookup ignoring answer from my address";
        return;
    }

    lock_t lock(mutex_);
    if (lookup_session_ && lookup_session_->id == msg.session_id)
    {
        if (msg.master || msg.up)
        {
            YLOG_L(info) << "lookup answer type=" << (msg.master ? "master" : "up")
                         << " from=" << addr << " session=" << msg.session_id;
            lookup_timer_->cancel();
            LookupCallback cb = lookup_session_->success_cb;
            lookup_session_.reset();
            lock.unlock();
            cb(false);
            return;
        }

        if (!msg.invalid)
        {
            YLOG_L(info) << "lookup answer type=normal from=" << addr
                         << " session=" << msg.session_id << " value=" << msg.value;
            lookup_session_->values[addr] = msg.value;
        }
        else
        {
            YLOG_L(info) << "lookup answer type=invalid from=" << addr
                         << " session=" << msg.session_id;
        }

        ++lookup_session_->answers;

        // prevent lookup finish if answers have been received at last try
        if (lookup_session_->prev_answers == 0)
        {
            YLOG_L(info) << "lookup retry counter reset";
            lookup_session_->retry_number = 1;
        }

        if (lookup_session_->answers == lookup_session_->cluster_size)
        {
            lookup_timer_->cancel();
            check_lookup_results(lock);
        }
    }
}

namespace {

inline bool values_less(
    const std::pair<netch_address, iid_t>& a,
    const std::pair<netch_address, iid_t>& b)
{
    return a.second < b.second;
}
}

void lookup_client::check_lookup_results(lock_t& lock)
{
    if (lookup_session_->answers == lookup_session_->cluster_size ||
        lookup_session_->retry_number == retry_count_)
    {
        if (lookup_session_->answers == 0)
        {
            ErrorCallback error_cb = lookup_session_->error_cb;
            lookup_session_.reset();
            lock.unlock();
            error_cb();
            return;
        }

        auto max = std::max_element(
            lookup_session_->values.begin(), lookup_session_->values.end(), values_less);
        iid_t my_value = lookup_session_->revision;
        if (max != lookup_session_->values.end())
        {
            YLOG_L(info) << "lookup session finished with max value " << max->second
                         << ", local value is " << my_value;
        }

        bool result =
            (lookup_session_->values.empty() ||
             (max != lookup_session_->values.end() && my_value >= max->second));
        LookupCallback success_cb = lookup_session_->success_cb;
        lookup_session_.reset();
        lock.unlock();
        success_cb(result);
    }
    else
    {
        lookup_try(lock);
        return;
    }
}

void lookup_client::lookup_try(lock_t& lock)
{
    lookup_session_->id = uuid_generator_();

    ++lookup_session_->retry_number;
    lookup_session_->values.clear();
    lookup_session_->prev_answers = lookup_session_->answers;
    lookup_session_->answers = 0;

    YLOG_L(info) << "lookup retry №" << lookup_session_->retry_number;

    lookup_message msg;
    msg.session_id = lookup_session_->id;
    lock.unlock();
    netch_->send_all(msg, sync_messages_.lookup);
    lookup_timer_->async_wait(timeout_, boost::bind(&lookup_client::check_lookup_deadline, this));
}

void lookup_client::check_lookup_deadline()

{
    YLOG_L(info) << "lookup timer";
    if (stop_)
    {
        return;
    }

    lock_t lock(mutex_);
    if (!lookup_session_)
    {
        return;
    }
    check_lookup_results(lock);
}

}
