#include "session.h"
#include <algorithm>
#include <boost/lambda/lambda.hpp>

std::atomic_size_t debugCounter{ 0 };
std::atomic_size_t sessionCounter{ 0 };

namespace ymod_pop_client {

session::session(
    yplatform::net::base_service* service,
    const yplatform::net::client_settings& settings)
    : parent_t(service, settings), resovle_timer_(*service->get_io()), debugId(debugCounter++)
{
    sessionCounter++;
    TASK_LOG(get_context(), debug)
        << "POP3 session#" << debugId << " created. Total: " << sessionCounter;
}

session::~session()
{
    sessionCounter--;
    TASK_LOG(get_context(), debug)
        << "POP3 session#" << debugId << " destroyed. Total: " << sessionCounter;
}

template <typename Buffer>
bool is_server_err(Buffer buff)
{
    boost::asio::streambuf::const_buffers_type cbufs = buff->data();
    boost::asio::buffers_iterator<boost::asio::streambuf::const_buffers_type> start =
        boost::asio::buffers_begin(cbufs);
    return *start != '+';
}

template <typename Buffer>
string get_response_info(Buffer buff)
{
    namespace l = boost::lambda;
    boost::asio::streambuf::const_buffers_type cbufs = buff->data();
    boost::asio::buffers_iterator<boost::asio::streambuf::const_buffers_type> start =
        boost::asio::buffers_begin(cbufs);
    boost::asio::buffers_iterator<boost::asio::streambuf::const_buffers_type> end =
        boost::asio::buffers_end(cbufs);
    return string(
        std::find_if(start, end, l::_1 == '\r' || l::_1 == '\n' || l::_1 == ' '),
        std::find_if(start, end, l::_1 == '\r' || l::_1 == '\n'));
}

template <typename Exception, typename Promise, typename Error, typename Buffer>
bool is_bad_result(Promise promise, Error err, Buffer buff)
{
    if (err)
    {
        buff->consume(buff->size());
        promise.set_exception(
            transport_error() << yplatform::system_error(err) << BOOST_ERROR_INFO);
    }
    else if (is_server_err(buff))
    {
        boost::asio::streambuf::const_buffers_type cbufs = buff->data();
        boost::asio::buffers_iterator<boost::asio::streambuf::const_buffers_type> start =
            boost::asio::buffers_begin(cbufs);
        boost::asio::buffers_iterator<boost::asio::streambuf::const_buffers_type> end =
            boost::asio::buffers_end(cbufs);
        end = std::find(start, end, '\n');
        string reason = get_response_info(buff);
        buff->consume(buff->size());
        promise.set_exception(
            Exception(reason) << yplatform::system_error(err) << server_response_info(reason)
                              << BOOST_ERROR_INFO);
    }
    else
        return false;
    return true;
}

future_string_ptr session::resolve(const std::string& server)
{
    promise_string_ptr out;
    auto self = shared_from_this();
    resolve(
        server,
        strand().wrap(boost::bind(&session::handle_resolve, self, out, _1, _2)),
        strand().wrap([out, server, this, self](auto ec) mutable {
            if (ec == boost::asio::error::operation_aborted) return;
            handle_timeout<promise_string_ptr>(out, "RESOLVE " + server);
        }));
    return out;
}

future_connect_result session::connect(const string& server, unsigned port, bool ssl)
{
    promise_connect_result out;
    parent_t::connect(
        server,
        port,
        boost::bind(&session::handle_connect, shared_from_this(), out, _1, ssl),
        boost::bind(
            &session::handle_timeout<promise_connect_result>, shared_from_this(), out, "CONNECT"));
    return out;
}

future_bool_t session::login(const string& user, const std::string& pass)
{
    promise_bool_t out;
    string request = "USER " + user + "\r\n";
    send_and_recv(
        request,
        boost::bind(&session::handle_user, shared_from_this(), out, pass, _1, _2),
        boost::bind(&session::handle_timeout<promise_bool_t>, shared_from_this(), out, "USER"),
        buffer_);
    return out;
}

future_string_ptr session::load_uidl(int id)
{
    filter_.set_mode(false);
    promise_string_ptr out;
    string request = "UIDL " + boost::lexical_cast<string>(id) + "\r\n";
    send_and_recv(
        request,
        boost::bind(&session::handle_current_uidl, shared_from_this(), out, id, _1, _2),
        boost::bind(
            &session::handle_timeout<promise_string_ptr>, shared_from_this(), out, "UIDL <id>"),
        buffer_);
    return out;
}

future_msg_list_ptr session::load_msg_list(bool size, bool uidls)
{
    filter_.set_mode(true);
    promise_msg_list_ptr out;
    message_list_ptr lst(new message_list_t());
    if (size)
    {
        send_and_recv(
            "LIST\r\n",
            boost::bind(&session::handle_list, shared_from_this(), out, lst, _1, _2, uidls),
            boost::bind(
                &session::handle_timeout<promise_msg_list_ptr>, shared_from_this(), out, "LIST"),
            buffer_);
    }
    else
    {
        send_and_recv(
            "UIDL\r\n",
            boost::bind(&session::handle_uidl, shared_from_this(), out, lst, _1, _2),
            boost::bind(
                &session::handle_timeout<promise_msg_list_ptr>, shared_from_this(), out, "UIDL"),
            buffer_);
    }
    return out;
}

future_string_ptr session::load_msg(int id, int lines, bool erase_dots)
{
    filter_.set_mode(true);
    promise_string_ptr out;
    string request;
    if (lines == -1) request = "RETR " + boost::lexical_cast<string>(id) + "\r\n";
    else
        request = "TOP " + boost::lexical_cast<string>(id) + " " +
            boost::lexical_cast<string>(lines) + "\r\n";

    send_and_recv(
        request,
        boost::bind(&session::handle_retr, shared_from_this(), out, _1, _2, erase_dots),
        boost::bind(
            &session::handle_timeout<promise_string_ptr>,
            shared_from_this(),
            out,
            request.substr(0, 4)),
        buffer_);
    return out;
}

future_mb_stat_ptr session::load_stat()
{
    filter_.set_mode(false);
    promise_mb_stat_ptr out;
    send_and_recv(
        "STAT\r\n",
        boost::bind(&session::handle_stat, shared_from_this(), out, _1, _2),
        boost::bind(&session::handle_timeout<promise_mb_stat_ptr>, shared_from_this(), out, "STAT"),
        buffer_);
    return out;
}

future_bool_t session::quit()
{
    filter_.set_mode(false);
    promise_bool_t out;
    send_and_recv(
        "QUIT\r\n",
        boost::bind(&session::handle_quit, shared_from_this(), out, _1, _2),
        boost::bind(&session::handle_timeout<promise_bool_t>, shared_from_this(), out, "QUIT"),
        buffer_);
    return out;
}

future_bool_t session::reset_changes()
{
    filter_.set_mode(false);
    promise_bool_t out;
    send_and_recv(
        "RSET\r\n",
        boost::bind(&session::handle_rset, shared_from_this(), out, _1, _2),
        boost::bind(&session::handle_timeout<promise_bool_t>, shared_from_this(), out, "RSET"),
        buffer_);
    return out;
}

future_bool_t session::delete_msg(int id)
{
    filter_.set_mode(false);
    promise_bool_t out;
    string request = "DELE " + boost::lexical_cast<string>(id) + "\r\n";
    send_and_recv(
        request,
        boost::bind(&session::handle_dele, shared_from_this(), out, _1, _2),
        boost::bind(&session::handle_timeout<promise_bool_t>, shared_from_this(), out, "DELE"),
        buffer_);
    return out;
}

void session::handle_resolve(
    promise_string_ptr out,
    const error_code& err,
    const boost::asio::ip::address& addr)
{
    if (err)
    {
        out.set_exception(resolve_error() << yplatform::system_error(err) << BOOST_ERROR_INFO);
    }
    else
    {
        out.set(string_ptr(new string(addr.to_string())));
    }
}

void session::handle_connect(promise_connect_result out, const error_code& err, bool ssl)
{
    if (err)
    {
        out.set_exception(connect_error() << yplatform::system_error(err) << BOOST_ERROR_INFO);
        return;
    }
    if (ssl)
    {
        start_tls(
            boost::bind(&session::handle_ssl, shared_from_this(), out, _1),
            boost::bind(
                &session::handle_timeout<promise_connect_result>, shared_from_this(), out, "TLS"));
    }
    else
    {
        async_read_until(
            boost::bind(&session::handle_welcome, shared_from_this(), out, _1, _2),
            boost::bind(
                &session::handle_timeout<promise_connect_result>,
                shared_from_this(),
                out,
                "WELCOME"),
            *buffer_,
            filter_);
    }
}

void session::handle_ssl(promise_connect_result out, const error_code& err)
{
    if (err)
    {
        out.set_exception(ssl_error() << yplatform::system_error(err) << BOOST_ERROR_INFO);
    }
    else
    {
        async_read_until(
            boost::bind(&session::handle_welcome, shared_from_this(), out, _1, _2),
            boost::bind(
                &session::handle_timeout<promise_connect_result>,
                shared_from_this(),
                out,
                "WELCOME"),
            *buffer_,
            filter_);
    }
}

void session::handle_welcome(
    promise_connect_result out,
    const error_code& err,
    std::size_t /*size*/)
{
    if (!is_bad_result<server_response_error>(out, err, buffer_))
    {
        auto buf_beg = boost::asio::buffers_begin(buffer_->data());
        std::string res{ buf_beg, buf_beg + buffer_->size() };
        buffer_->consume(buffer_->size());
        error_code err_code;
        out.set({ remote_addr().to_string(err_code), std::move(res) });
    }
    else
    {
        NET_SESSION_LOG(error) << "error in handle_welcome: " << err.message();
    }
}

void session::handle_user(
    promise_bool_t out,
    const string& pass,
    const error_code& err,
    std::size_t /*size*/)
{
    if (!is_bad_result<server_response_error>(out, err, buffer_))
    {
        buffer_->consume(buffer_->size());
        string request = "PASS " + pass + "\r\n";
        send_and_recv(
            request,
            boost::bind(&session::handle_pass, shared_from_this(), out, _1, _2),
            boost::bind(&session::handle_timeout<promise_bool_t>, shared_from_this(), out, "PASS"),
            buffer_);
    }
    else
    {
        NET_SESSION_LOG(error) << "error in handle_user: " << err.message();
    }
}

void session::handle_pass(promise_bool_t out, const error_code& err, std::size_t /*size*/)
{
    if (!is_bad_result<login_error>(out, err, buffer_))
    {
        buffer_->consume(buffer_->size());
        out.set(true);
    }
    else
    {
        if (err) NET_SESSION_LOG(error) << "error in handle_pass: " << err.message();
    }
}

void session::handle_current_uidl(
    promise_string_ptr out,
    int /*id*/,
    const error_code& err,
    std::size_t /*size*/)
{
    if (is_bad_result<server_response_error>(out, err, buffer_))
    {
        NET_SESSION_LOG(error) << "error in handle_uidl<id>: " << err.message();
        return;
    }
    string_ptr message_uidl(new string);
    try
    {
        typedef boost::asio::buffers_iterator<boost::asio::streambuf::const_buffers_type> iter_t;
        boost::asio::streambuf::const_buffers_type cbufs = buffer_->data();
        iter_t buff_beg = boost::asio::buffers_begin(cbufs);
        iter_t buff_end = boost::asio::buffers_end(cbufs);
        iter_t id_begin = std::find(buff_beg, buff_end, ' ');
        if (id_begin == buff_end) throw std::runtime_error("invalid response format");
        ++id_begin;
        iter_t uidl_begin = std::find(id_begin, buff_end, ' ');
        if (uidl_begin == buff_end) throw std::runtime_error("invalid response format");
        // TODO : add id check
        ++uidl_begin;
        iter_t uidl_end = uidl_begin;
        while (uidl_end != buff_end && *uidl_end != '\r' && *uidl_end != '\n')
            ++uidl_end;
        message_uidl->append(uidl_begin, uidl_end);
    }
    catch (...)
    {
        NET_SESSION_LOG(error) << "error in handle_uidl<id>: format exception";
        buffer_->consume(buffer_->size());
        out.set_exception(server_format_error() << BOOST_ERROR_INFO);
        return;
    }
    out.set(message_uidl);
}

void session::handle_list(
    promise_msg_list_ptr out,
    message_list_ptr lst,
    const error_code& err,
    std::size_t /*size*/,
    bool uidls)
{
    if (is_bad_result<server_response_error>(out, err, buffer_))
    {
        NET_SESSION_LOG(error) << "error in handle_list: " << err.message();
        return;
    }
    std::istream stream(buffer_.get());
    std::string line;
    std::getline(stream, line);
    while (stream.good())
    {
        std::getline(stream, line);
        if (line[0] == '.') // end list
            break;
        try
        {
            std::size_t delim = line.find(' ', 0);
            int id = boost::lexical_cast<int>(line.substr(0, delim));
            std::size_t size;
            if (line[line.size() - 1] == '\r')
                size = boost::lexical_cast<int>(line.substr(delim + 1, line.size() - delim - 2));
            else
                size = boost::lexical_cast<int>(line.substr(delim + 1));

            (*lst)[id].id = id;
            (*lst)[id].size = size;
        }
        catch (...)
        {
            NET_SESSION_LOG(error) << "error in handle_list: format exception";
            buffer_->consume(buffer_->size());
            out.set_exception(server_format_error() << BOOST_ERROR_INFO);
            return;
        }
    }
    if (uidls)
    {
        filter_.set_mode(true);
        send_and_recv(
            "UIDL\r\n",
            boost::bind(&session::handle_uidl, shared_from_this(), out, lst, _1, _2),
            boost::bind(
                &session::handle_timeout<promise_msg_list_ptr>, shared_from_this(), out, "UIDL"),
            buffer_);
    }
    else
    {
        out.set(lst);
    }
}

void session::handle_uidl(
    promise_msg_list_ptr out,
    message_list_ptr lst,
    const error_code& err,
    std::size_t /*size*/)
{
    if (is_bad_result<server_response_error>(out, err, buffer_))
    {
        NET_SESSION_LOG(error) << "error in handle_uidl: " << err.message();
        return;
    }
    std::istream stream(buffer_.get());
    std::string line;
    std::getline(stream, line);
    while (stream.good())
    {
        std::getline(stream, line);
        if (line[0] == '.') // end list
            break;
        try
        {
            std::size_t delim = line.find(' ', 0);
            int id = boost::lexical_cast<int>(line.substr(0, delim));
            std::size_t uidl_chunk_size = line.npos;
            if (line[line.size() - 1] == '\r') uidl_chunk_size = line.size() - delim - 2;

            (*lst)[id].id = id;
            (*lst)[id].uidl = line.substr(delim + 1, uidl_chunk_size);
        }
        catch (...)
        {
            NET_SESSION_LOG(error) << "error in handle_uidl: format exception";
            buffer_->consume(buffer_->size());
            out.set_exception(server_format_error() << BOOST_ERROR_INFO);
            return;
        }
    }
    out.set(lst);
}

void session::handle_retr(
    promise_string_ptr out,
    const error_code& err,
    std::size_t /*size*/,
    bool erase_dots)
{
    if (is_bad_result<server_response_error>(out, err, buffer_))
    {
        NET_SESSION_LOG(error) << "error in handle_retr: " << err.message();
        return;
    }
    string_ptr res(new string);
    try
    {
        boost::asio::streambuf::const_buffers_type cbufs = buffer_->data();
        typedef boost::asio::buffers_iterator<boost::asio::streambuf::const_buffers_type> iter_t;
        iter_t buff_beg = boost::asio::buffers_begin(cbufs);
        iter_t buff_end = boost::asio::buffers_end(cbufs);
        iter_t i = std::find(buff_beg, buff_end, '\n');
        if (i == buff_end) throw std::runtime_error("unknown format");
        ++i;
        if (erase_dots)
        {
            iter_t i_dot = i;
            res->reserve(buff_end - i);
            while ((i_dot = std::find(i_dot, buff_end, '\n')) != buff_end)
            {
                if (++i_dot == buff_end) break;
                if (*i_dot != '.') continue;
                ++i_dot;
                if (i_dot != buff_end && *i_dot == '.')
                {
                    std::copy(i, i_dot, std::back_insert_iterator<string>(*res));
                    i = i_dot + 1;
                }
                else
                {
                    // finish dot in multiline response
                    buff_end = --i_dot;
                    break;
                }
            }
            std::copy(i, buff_end, std::back_insert_iterator<string>(*res));
        }
        else
        {
            std::copy(i, buff_end, std::back_insert_iterator<string>(*res));
        }
    }
    catch (...)
    {
        NET_SESSION_LOG(error) << "error in handle_retr: format exception";
        buffer_->consume(buffer_->size());
        out.set_exception(server_format_error() << BOOST_ERROR_INFO);
        return;
    }
    buffer_->consume(buffer_->size());
    out.set(res);
}

void session::handle_dele(promise_bool_t out, const error_code& err, std::size_t /*size*/)
{
    if (is_bad_result<server_response_error>(out, err, buffer_))
    {
        NET_SESSION_LOG(error) << "error in handle_dele: " << err.message();
        return;
    }
    buffer_->consume(buffer_->size());
    out.set(true);
}

void session::handle_rset(promise_bool_t out, const error_code& err, std::size_t /*size*/)
{
    if (is_bad_result<server_response_error>(out, err, buffer_))
    {
        NET_SESSION_LOG(error) << "error in handle_rset: " << err.message();
        return;
    }
    buffer_->consume(buffer_->size());
    out.set(true);
}

void session::handle_stat(promise_mb_stat_ptr out, const error_code& err, std::size_t /*size*/)
{
    if (is_bad_result<server_response_error>(out, err, buffer_))
    {
        NET_SESSION_LOG(error) << "error in handle_stat: " << err.message();
        return;
    }
    int count = -1;
    std::size_t all_size = 0;
    try
    {
        typedef boost::asio::buffers_iterator<boost::asio::streambuf::const_buffers_type> iter_t;
        boost::asio::streambuf::const_buffers_type cbufs = buffer_->data();
        iter_t buff_beg = boost::asio::buffers_begin(cbufs);
        iter_t buff_end = boost::asio::buffers_end(cbufs);
        iter_t count_begin = std::find(buff_beg, buff_end, ' ');
        if (count_begin == buff_end) throw std::runtime_error("invalid response format");
        ++count_begin;
        iter_t size_begin = std::find(count_begin, buff_end, ' ');
        if (size_begin == buff_end) throw std::runtime_error("invalid response format");
        count = boost::lexical_cast<int>(string(count_begin, size_begin));
        ++size_begin;
        iter_t size_end = size_begin;
        while (size_end != buff_end && std::isdigit(*size_end))
            ++size_end;
        all_size = boost::lexical_cast<std::size_t>(string(size_begin, size_end));
    }
    catch (...)
    {
        NET_SESSION_LOG(error) << "error in handle_stat: format exception";
        buffer_->consume(buffer_->size());
        out.set_exception(server_format_error() << BOOST_ERROR_INFO);
        return;
    }
    out.set(mailbox_stat_ptr(new mailbox_stat(count, all_size)));
}

void session::handle_quit(promise_bool_t out, const error_code& err, std::size_t /*size*/)
{
    if (is_bad_result<server_response_error>(out, err, buffer_))
    {
        NET_SESSION_LOG(error) << "error in handle_quit: " << err.message();
        return;
    }
    buffer_->consume(buffer_->size());
    out.set(true);
}

void session::resolve(
    const string& addr,
    resolve_handler_t handler,
    timer_handler_t deadline_handler)
{
    typedef yplatform::net::dns::resolver Resolver;
    typedef yplatform::net::detail::resolve_step_helper<yplatform::net::ipv4, Resolver>
        Resolve4Step;
    typedef yplatform::net::detail::resolve_step_helper<yplatform::net::ipv6, Resolver>
        Resolve6Step;

    error_code errorCode;
    auto ip = boost::asio::ip::address::from_string(addr, errorCode);
    if (!errorCode)
    {
        handler(errorCode, ip);
    }
    else
    {
        resovle_timer_.expires_from_now(settings().resolve_timeout);
        resovle_timer_.async_wait(deadline_handler);
        if (settings().resolve_order == yplatform::net::client_settings::ipv4)
        {
            Resolve4Step::resolve(
                get_resolver(),
                addr,
                boost::bind(
                    &session::handle_resolve_internal<typename Resolve4Step::iterator>,
                    this,
                    _1,
                    _2,
                    handler,
                    addr));
        }
        else if (settings().resolve_order == yplatform::net::client_settings::ipv6)
        {
            Resolve6Step::resolve(
                get_resolver(),
                addr,
                boost::bind(
                    &session::handle_resolve_internal<typename Resolve6Step::iterator>,
                    this,
                    _1,
                    _2,
                    handler,
                    addr));
        }
        else
        {
            throw std::runtime_error("not implemented");
        }
    }
}

}
