#ifndef _YMOD_POP_CLIENT_SESSION_H_
#define _YMOD_POP_CLIENT_SESSION_H_

#include <boost/enable_shared_from_this.hpp>
#include <yplatform/future/future.hpp>
#include <yplatform/net/client_session_strand.h>
#include <ymod_popclient/call.h>
#include <ymod_popclient/errors.h>
#include "filter.h"
#include <map>

namespace ymod_pop_client {

class session
    : public yplatform::net::client_session_strand<>
    , public boost::enable_shared_from_this<session>
{
    using stream_buffer = boost::asio::basic_streambuf<std::allocator<char>>;
    using stream_buffer_ptr = std::shared_ptr<stream_buffer>;
    using timer = yplatform::time_traits::timer;

    using resolve_handler_t =
        std::function<void(const boost::system::error_code&, const boost::asio::ip::address&)>;
    using timer_handler_t = std::function<void(const boost::system::error_code&)>;

public:
    typedef yplatform::net::client_session_strand<> parent_t;

    session(yplatform::net::base_service* service, const yplatform::net::client_settings& settings);
    ~session();

    future_string_ptr resolve(const std::string& server);
    future_connect_result connect(const std::string& server, unsigned port, bool ssl);
    future_bool_t login(const std::string& user, const std::string& pass);
    future_string_ptr load_uidl(int id);
    future_msg_list_ptr load_msg_list(bool size, bool uidls);
    future_string_ptr load_msg(int id, int lines, bool erase_dots);
    future_mb_stat_ptr load_stat();
    future_bool_t quit();
    future_bool_t reset_changes();
    future_bool_t delete_msg(int id);

    void set_context(yplatform::task_context_ptr task_context)
    {
        if (task_context)
        {
            TASK_LOG(task_context, debug)
                << "POP3 session#" << debugId << " context set:" << task_context->uniq_id();
        }
        task_context_ = task_context;
    }

    virtual yplatform::task_context_ptr get_context() const
    {
        return task_context_.lock();
    }

private:
    typedef boost::system::error_code error_code;

    template <typename Handler, typename DeadlineHandler>
    void send_and_recv(
        const std::string& command_text,
        Handler&& handler,
        DeadlineHandler&& deadline_handler,
        stream_buffer_ptr recv_buffer)
    {
        auto send_bufer = std::make_shared<stream_buffer>();
        send_bufer->sputn(command_text.data(), command_text.size());
        auto send_handler =
            [handler, deadline_handler, recv_buffer, this, self = shared_from_this()](
                const error_code& err, std::size_t /* size */, stream_buffer_ptr /* buff */) {
                if (err)
                {
                    return handler(err, 0, buffer_);
                }
                async_read_until(handler, deadline_handler, *recv_buffer, filter_);
            };
        async_write_shared_buff(send_bufer, send_handler, deadline_handler);
    }

    void resolve(const string& addr, resolve_handler_t handler, timer_handler_t deadline_handler);

    template <typename Iterator>
    void handle_resolve_internal(
        const error_code& e,
        Iterator i,
        resolve_handler_t handler,
        const string& addr);

    void handle_resolve(
        promise_string_ptr out,
        const error_code& err,
        const boost::asio::ip::address& addr);

    void handle_connect(promise_connect_result out, const error_code& err, bool ssl);

    void handle_ssl(promise_connect_result out, const error_code& err);

    void handle_welcome(promise_connect_result out, const error_code& err, std::size_t size);

    void handle_user(
        promise_bool_t out,
        const string& pass,
        const error_code& err,
        std::size_t size);

    void handle_pass(promise_bool_t out, const error_code& err, std::size_t size);

    void handle_current_uidl(
        promise_string_ptr out,
        int id,
        const error_code& err,
        std::size_t size);

    void handle_list(
        promise_msg_list_ptr out,
        message_list_ptr lst,
        const error_code& err,
        std::size_t size,
        bool uidls);

    void handle_uidl(
        promise_msg_list_ptr out,
        message_list_ptr lst,
        const error_code& err,
        std::size_t size);

    void handle_retr(
        promise_string_ptr out,
        const error_code& err,
        std::size_t size,
        bool erase_dots);

    void handle_dele(promise_bool_t out, const error_code& err, std::size_t size);

    void handle_rset(promise_bool_t out, const error_code& err, std::size_t size);

    void handle_stat(promise_mb_stat_ptr out, const error_code& err, std::size_t size);

    void handle_quit(promise_bool_t out, const error_code& err, std::size_t size);

    template <typename Promise>
    void handle_timeout(Promise out, const string& command);

    timer resovle_timer_;
    stream_buffer_ptr buffer_ = std::make_shared<stream_buffer>();
    ymod_pop_client::filter filter_;
    yplatform::task_context_weak_ptr task_context_;
    size_t debugId = 0;
};

template <typename Promise>
void session::handle_timeout(Promise out, const string& command)
{
    NET_SESSION_LOG(error) << "error in handle_timeout: timeout expire. command: " << command;
    try
    {
        cancel_operations();
    }
    catch (...)
    {
    }
    out.set_exception(connection_timeout() << BOOST_ERROR_INFO);
}

template <typename Iterator>
void session::handle_resolve_internal(
    const error_code& e,
    Iterator i,
    resolve_handler_t handler,
    const string& addr)
{
    resovle_timer_.cancel();
    if (e)
    {
        NET_SESSION_LOG(error) << "handleResolve: error: " << e.message() << ", addr: " << addr;
        handler(e, boost::asio::ip::address());
    }
    else if (i == Iterator())
    {
        NET_SESSION_LOG(error) << "handleResolve: error: empty resolve list"
                               << ", addr: " << addr;
        handler(e, boost::asio::ip::address());
    }
    else
    {
        handler(error_code(), boost::asio::ip::address::from_string(*i));
    }
}

typedef boost::shared_ptr<session> session_ptr;

}

#endif // _YMOD_POP_CLIENT_SESSION_H_
