#ifndef _YMOD_PQ_BIND_ARRAY_H_
#define _YMOD_PQ_BIND_ARRAY_H_

#include <vector>
#include <boost/shared_ptr.hpp>
#include <boost/date_time.hpp>

namespace ymod_pq {

struct base_bind_value
{
    virtual ~base_bind_value()
    {
    }
    virtual std::string value_string() const = 0;
};

typedef boost::shared_ptr<base_bind_value> bind_value_ptr;

template <typename ValueT>
struct impl_bind_value : public base_bind_value
{
    virtual const ValueT& value() const = 0;
    virtual std::string value_string() const
    {
        std::ostringstream os;
        os << value();
        return os.str();
    }
};

template <typename T>
struct impl_bind_value<std::vector<T>> : public base_bind_value
{
    virtual const std::vector<T>& value() const = 0;
    virtual std::string value_string() const
    {
        return std::string();
    }
};

template <typename ValueT>
struct cref_bind_value : public impl_bind_value<ValueT>
{
    cref_bind_value(const ValueT& value_) : val(value_)
    {
    }

    const ValueT& value() const
    {
        return val;
    }

    const ValueT& val;
};

template <typename T>
struct cref_bind_value<std::vector<T>> : public impl_bind_value<std::vector<T>>
{
    cref_bind_value(const std::vector<T>& value_) : val(value_)
    {
    }

    const std::vector<T>& value() const
    {
        return val;
    }

    const std::vector<T>& val;
};

template <typename ValueT>
struct const_bind_value : public impl_bind_value<ValueT>
{
    const_bind_value(const ValueT& value_) : val(value_)
    {
    }

    const ValueT& value() const
    {
        return val;
    }

    ValueT val;
};

template <typename T>
struct const_bind_value<std::vector<T>> : public impl_bind_value<std::vector<T>>
{
    const_bind_value(const std::vector<T>& value_) : val(value_)
    {
    }

    const std::vector<T>& value() const
    {
        return val;
    }

    std::vector<T> val;
};

template <typename ValueT>
struct swap_bind_value : public impl_bind_value<ValueT>
{
    swap_bind_value(ValueT& value_)
    {
        val.swap(value_);
    }

    const ValueT& value() const
    {
        return val;
    }

    ValueT val;
};

template <typename T>
struct swap_bind_value<std::vector<T>> : public impl_bind_value<std::vector<T>>
{
    swap_bind_value(std::vector<T>& value_)
    {
        val.swap(value_);
    }

    const std::vector<T>& value() const
    {
        return val;
    }

    std::vector<T> val;
};

struct cptr_bind_value : public base_bind_value
{
    cptr_bind_value(const void* v, int sz) : ptr_(v), sz_(sz)
    {
    }
    std::string value_string() const
    {
        return std::string();
    }
    const void* ptr_;
    int sz_;
};

class bind_array
{
public:
    enum TYPES
    {
        STRING_ARRAY,
        INT64_ARRAY,
        INT,
        UINT,
        FLOAT,
        DOUBLE,
        STRING,
        BYTES,
        DATE,
        TIMESTAMP,
        BLOB,
        CLOB,
        BFILE,
        REF,
        POBJECT,
        ORA_NULL,
        BYTE_ARRAY
    };

protected:
    template <typename Val1, typename Val2, typename Val3>
    struct three
    {
        three()
        {
        }
        three(const Val1& v1, const Val2& v2, const Val3& v3)
            : value_1(v1), value_2(v2), value_3(v3)
        {
        }
        Val1 value_1;
        Val2 value_2;
        Val3 value_3;
    };

    typedef three<TYPES, std::string, bind_value_ptr> bind_value_three;

public:
    void push_back(TYPES type, const bind_value_ptr& val)
    {
        values_.push_back(bind_value_three(type, "", val));
    }

    void push_named(TYPES type, const std::string& name, const bind_value_ptr& val)
    {
        values_.push_back(bind_value_three(type, name, val));
    }

    std::size_t size() const
    {
        return values_.size();
    }

    TYPES type(std::size_t i) const
    {
        return values_[i].value_1;
    }

    const std::string& name(std::size_t i) const
    {
        return values_[i].value_2;
    }

    template <typename ValueT>
    const ValueT& value(std::size_t i) const
    {
        return boost::dynamic_pointer_cast<impl_bind_value<ValueT>>(values_[i].value_3)->value();
    }

    std::string value_string(std::size_t i) const
    {
        base_bind_value* v = values_[i].value_3.get();
        return v->value_string();
    }

    template <typename Holder>
    boost::shared_ptr<Holder> value_holder(std::size_t i) const
    {
        return boost::static_pointer_cast<Holder>(values_[i].value_3);
    }

private:
    std::vector<bind_value_three> values_;
};

typedef boost::shared_ptr<bind_array> bind_array_ptr;

template <typename ValueT>
inline bind_value_ptr make_cref_bind_value(const ValueT& value)
{
    return bind_value_ptr(new cref_bind_value<ValueT>(value));
}

template <typename ValueT>
inline bind_value_ptr make_const_bind_value(const ValueT& value)
{
    return bind_value_ptr(new const_bind_value<ValueT>(value));
}

template <typename ValueT>
inline bind_value_ptr make_swap_bind_value(ValueT& value)
{
    return bind_value_ptr(new swap_bind_value<ValueT>(value));
}

inline bind_value_ptr make_cptr_bind_value(const void* v, int sz)
{
    return bind_value_ptr(new cptr_bind_value(v, sz));
}

inline void push_swap_string(bind_array_ptr arr, std::string& v)
{
    arr->push_back(bind_array::STRING, make_swap_bind_value(v));
}

inline void push_swap_svector(bind_array_ptr arr, std::vector<std::string>& v)
{
    arr->push_back(bind_array::STRING_ARRAY, make_swap_bind_value(v));
}

inline void push_swap_int64_vector(bind_array_ptr arr, std::vector<int64_t>& v)
{
    arr->push_back(bind_array::INT64_ARRAY, make_swap_bind_value(v));
}

//*****************************************************************************

inline void push_const_uint(bind_array_ptr arr, unsigned int v)
{
    arr->push_back(bind_array::UINT, make_const_bind_value(v));
}

inline void push_const_int(bind_array_ptr arr, int v)
{
    arr->push_back(bind_array::INT, make_const_bind_value(v));
}

inline void push_const_float(bind_array_ptr arr, float v)
{
    arr->push_back(bind_array::FLOAT, make_const_bind_value(v));
}

inline void push_const_double(bind_array_ptr arr, double v)
{
    arr->push_back(bind_array::DOUBLE, make_const_bind_value(v));
}

inline void push_const_string(bind_array_ptr arr, const std::string& v)
{
    arr->push_back(bind_array::STRING, make_const_bind_value(v));
}

inline void push_const_svector(bind_array_ptr arr, const std::vector<std::string>& v)
{
    arr->push_back(bind_array::STRING_ARRAY, make_const_bind_value(v));
}

inline void push_const_int64_vector(bind_array_ptr arr, const std::vector<int64_t>& v)
{
    arr->push_back(bind_array::INT64_ARRAY, make_const_bind_value(v));
}

inline void push_const_date(bind_array_ptr arr, const boost::posix_time::ptime& v)
{
    arr->push_back(bind_array::DATE, make_const_bind_value(v));
}

inline void push_const_time_t(bind_array_ptr arr, std::time_t v)
{
    arr->push_back(bind_array::DATE, make_const_bind_value(boost::posix_time::from_time_t(v)));
}

inline void push_null(bind_array_ptr arr, bind_array::TYPES t)
{
    arr->push_back(bind_array::ORA_NULL, make_const_bind_value(int(t)));
}
//*****************************************************************************

inline void push_cref_string(bind_array_ptr arr, const std::string& v)
{
    arr->push_back(bind_array::STRING, make_cref_bind_value(v));
}

inline void push_cref_svector(bind_array_ptr arr, const std::vector<std::string>& v)
{
    arr->push_back(bind_array::STRING_ARRAY, make_cref_bind_value(v));
}

inline void push_cref_int64_vector(bind_array_ptr arr, const std::vector<int64_t>& v)
{
    arr->push_back(bind_array::INT64_ARRAY, make_cref_bind_value(v));
}

inline void push_cptr_byte_array(bind_array_ptr arr, const void* v, int sz)
{
    arr->push_back(bind_array::BYTE_ARRAY, make_cptr_bind_value(v, sz));
}

} // namespace ymod_pq

#endif // _YMOD_PQ_BIND_ARRAY_H_
