#pragma once

#include "settings.h"
#include "basic_cluster.h"
#include "replica_monitor.h"
#include "pq_impl.h"
#include <yplatform/module.h>
#include <yplatform/find.h>
#include <yplatform/hash/md5.h>

#include <mutex>
#include <unordered_map>

namespace ymod_pq {

class cluster_impl
    : public cluster_call
    , public yplatform::module
{
    using cluster = basic_cluster<call_impl, replica_monitor>;
    using cluster_ptr = std::shared_ptr<cluster>;

public:
    using host_state = cluster::host_state;

    struct settings
    {
        cluster::settings cluster;
        replica_monitor::settings replica_monitor;
        std::string monitor_pq;

        settings() = default;
        settings(const yplatform::ptree& xml)
        {
            using yplatform::time_traits::duration;

            cluster.master.parse(xml.get_child("master"));
            cluster.replica.parse(xml.get_child("replica"));
            cluster.fallback.parse(xml.get_child("fallback"));
            cluster.enable_fallback_error_rate = xml.get<double>("enable_fallback_error_rate");
            cluster.disable_fallback_error_rate = xml.get<double>("disable_fallback_error_rate");
            cluster.probe_rps = xml.get<double>("probe_rps");
            cluster.rate_accumulator_window = xml.get<unsigned>("rate_accumulator_window");
            cluster.auto_fallback_enabled = xml.get<bool>("auto_fallback_enabled");
            replica_monitor.query = xml.get<std::string>("replica_monitor.query");
            replica_monitor.poll_interval = xml.get<duration>("replica_monitor.poll_interval");
            replica_monitor.replica_list_ttl =
                xml.get<duration>("replica_monitor.replica_list_ttl");
            monitor_pq = xml.get<std::string>("monitor_pq");
        }
    };

    cluster_impl(yplatform::reactor& reactor, const yplatform::ptree& xml)
        : reactor_(reactor)
        , settings_(xml)
        , monitor_pq_(yplatform::find<call, std::shared_ptr>(settings_.monitor_pq))
    {
    }

    void stop()
    {
        for (auto& [db, cluster] : clusters_)
        {
            cluster->stop();
        }
    }

    virtual future_result request(
        yplatform::task_context_ptr ctx,
        const std::string& db,
        const std::string& request,
        bind_array_ptr bind_vars,
        response_handler_ptr handler,
        bool log_timings,
        const yplatform::time_traits::duration& deadline,
        request_target target) override
    {
        return get_cluster(db)->request(
            ctx, request, bind_vars, handler, log_timings, deadline, target);
    }

    virtual future_up_result update(
        yplatform::task_context_ptr ctx,
        const std::string& db,
        const std::string& request,
        bind_array_ptr bind_arr,
        bool log_timings,
        const yplatform::time_traits::duration& deadline,
        request_target target) override
    {
        return get_cluster(db)->update(ctx, request, bind_arr, log_timings, deadline, target);
    }

    virtual future_result execute(
        yplatform::task_context_ptr ctx,
        const std::string& db,
        const std::string& request,
        bind_array_ptr bind_arr,
        bool log_timings,
        const yplatform::time_traits::duration& deadline,
        request_target target) override
    {
        return get_cluster(db)->execute(ctx, request, bind_arr, log_timings, deadline, target);
    }

    yplatform::ptree get_stats() const override
    {
        std::lock_guard lock(mutex_);
        yplatform::ptree ret;
        for (auto& [db, cluster] : clusters_)
        {
            auto cluster_stats = cluster->get_stats();
            ret.put_child(yplatform::md5(db), cluster_stats);
        }
        return ret;
    }

    virtual cluster_health_by_db get_health() override
    {
        std::lock_guard lock(mutex_);
        cluster_health_by_db ret;
        for (auto& [db, cluster] : clusters_)
        {
            ret[db] = cluster->get_health();
        }
        return ret;
    }

    virtual void enable_auto_fallback(const yplatform::task_context_ptr& ctx) override
    {
        std::lock_guard lock(mutex_);
        control_auto_fallback(ctx, true);
    }

    virtual void disable_auto_fallback(const yplatform::task_context_ptr& ctx) override
    {
        std::lock_guard lock(mutex_);
        control_auto_fallback(ctx, false);
    }

    virtual void enable_fallback(
        const yplatform::task_context_ptr& ctx,
        const std::string& db,
        request_target t) override
    {
        auto maybe_cluster = get_cluster_optional(db);
        if (maybe_cluster)
        {
            (*maybe_cluster)->enable_fallback(ctx, t);
        }
    }

    virtual void disable_fallback(
        const yplatform::task_context_ptr& ctx,
        const std::string& db,
        request_target t) override
    {
        auto maybe_cluster = get_cluster_optional(db);
        if (maybe_cluster)
        {
            (*maybe_cluster)->disable_fallback(ctx, t);
        }
    }

private:
    cluster_ptr get_cluster(const std::string& db)
    {
        std::lock_guard lock(mutex_);
        auto it = clusters_.find(db);
        if (it == clusters_.end())
        {
            it = clusters_.insert(std::make_pair(db, make_cluster(db))).first;
        }
        return it->second;
    }

    std::optional<cluster_ptr> get_cluster_optional(const std::string& db)
    {
        std::lock_guard lock(mutex_);
        auto it = clusters_.find(db);
        return it != clusters_.end() ? it->second : std::optional<cluster_ptr>{};
    }

    cluster_ptr make_cluster(const std::string& db)
    {
        auto make_pq = [reactor = &reactor_,
                        logger = this->logger()](const ymod_pq::settings& settings) {
            auto pq = std::make_shared<call_impl>(*reactor, settings);
            pq->logger(logger);
            return pq;
        };
        auto monitor_ctx = boost::make_shared<yplatform::task_context>();
        auto monitor = std::make_shared<replica_monitor>(
            monitor_ctx, logger(), *reactor_.io(), db, monitor_pq_, settings_.replica_monitor);
        auto new_cluster = std::make_shared<cluster>(db, make_pq, monitor, settings_.cluster);
        new_cluster->start();
        return new_cluster;
    }

    void control_auto_fallback(const yplatform::task_context_ptr& ctx, bool enable)
    {
        settings_.cluster.auto_fallback_enabled = enable;
        for (auto& [db, cluster] : clusters_)
        {
            if (enable)
            {
                cluster->enable_auto_fallback(ctx);
            }
            else
            {
                cluster->disable_auto_fallback(ctx);
            }
        }
    }

    yplatform::reactor& reactor_;
    settings settings_;
    call_ptr monitor_pq_;

    mutable std::mutex mutex_;
    std::unordered_map<std::string, cluster_ptr> clusters_;
};

}
