#pragma once

#include <yplatform/util/split.h>
#include <unordered_map>
#include <libpq-fe.h>

namespace ymod_pq {

class conninfo
{
public:
    conninfo() = default;
    conninfo(const std::string& s)
    {
        char* libpq_parse_error = nullptr;
        auto conninfo_opts = std::unique_ptr<PQconninfoOption, void (*)(PQconninfoOption*)>(
            PQconninfoParse(s.c_str(), &libpq_parse_error), PQconninfoFree);
        auto libpq_error_manager =
            std::unique_ptr<char, void (*)(void*)>(libpq_parse_error, PQfreemem);
        if (!conninfo_opts)
        {
            std::string parse_error(libpq_parse_error ? libpq_parse_error : "unknown");
            throw std::runtime_error("failed to parse conninfo, error: " + parse_error);
        }
        // https://www.postgresql.org/docs/10/static/libpq-connect.html
        // According to the docs, conninfo_opts is a pointer
        // to an array of PQconninfoOption structs, the last
        // has keyword set to null.
        for (auto opt = conninfo_opts.get(); opt->keyword; ++opt)
        {
            if (opt->val)
            {
                params_[opt->keyword] = opt->val;
            }
        }
    }

    std::string to_string() const
    {
        using namespace std::string_literals;
        std::string ret;
        for (auto& key_val : params_)
        {
            ret += key_val.first + "="s + escape(key_val.second) + " "s;
        }
        if (ret.size())
        {
            ret.resize(ret.size() - 1);
        }
        return ret;
    }

    std::string get(const std::string& key) const
    {
        return params_.count(key) ? params_.at(key) : "";
    }

    std::vector<std::string> hosts() const
    {
        using namespace std::string_literals;
        using yplatform::util::split;

        return params_.count("host"s) > 0 ? split(params_.at("host"s), ","s) :
                                            std::vector<std::string>{};
    }

    void set(const std::string& key, const std::string& value)
    {
        params_[key] = value;
    }

    void reset(const std::string& key)
    {
        params_.erase(key);
    }

private:
    std::string escape(const std::string& value) const
    {
        auto ret = escape_quotes_backslashes(value);
        if (ret.empty() || ret.find(' ') != std::string::npos)
        {
            ret = '\'' + ret + '\'';
        }
        return ret;
    }

    std::string escape_quotes_backslashes(const std::string& s) const
    {
        std::string escaped;
        escaped.reserve(s.size() * 2);
        for (auto c : s)
        {
            if (c == '\'' || c == '\\')
            {
                escaped += '\\';
            }
            escaped += c;
        }
        return escaped;
    }

    using params = std::unordered_map<std::string, std::string>;

    params params_;
};

}