#include "pq_impl.h"
#include <pa/async.h>
#include <yplatform/net/dns/resolver_service.h>
#include <yplatform/util/split.h>
#include <yplatform/util/sstream.h>
#include <yplatform/find.h>
#include <yplatform/log.h>
#include <boost/algorithm/string/predicate.hpp>
#include <boost/chrono.hpp>
#include <type_traits>

static std::string log_db(const std::string& db)
{
    static const std::string HOST = "host";
    static const std::string DBNAME = "dbname";
    static const std::string SEPARATOR = " ";

    bool host_extracted = false;
    bool dbname_extracted = false;
    std::string db_for_log;
    yplatform::sstream db_stream(db_for_log, 128);
    auto db_splitted = yplatform::util::split(db, SEPARATOR);
    for (auto& param_kv : db_splitted)
    {
        if (boost::starts_with(param_kv, HOST))
        {
            // Truncate host, since this string is rather long
            // for pgaas clusters, but one unqualified hostname
            // is sufficient to identify the db.
            db_stream << param_kv.substr(0, param_kv.find('.')) << ' ';
            host_extracted = true;
        }
        else if (boost::starts_with(param_kv, DBNAME))
        {
            db_stream << param_kv << ' ';
            dbname_extracted = true;
        }
    }
    return host_extracted && dbname_extracted ? db_for_log : db;
}
namespace ymod_pq {

typedef std::shared_ptr<yplatform::log::contains_logger> contains_logger_ptr;

call_impl::~call_impl()
{
    YLOG_L(debug) << "pq_module task destroyed";
}

void call_impl::fini()
{
    try
    {
        clear();
    }
    catch (...)
    {
    }
}

namespace {

class request_profiler
{
public:
    request_profiler(
        yplatform::task_context_ptr ctx,
        const std::string& name,
        const std::string& db,
        bool timings,
        bool pa,
        contains_logger_ptr contains_logger)
        : ctx_(ctx)
        , name_(name)
        , db_(db)
        , log_timings(timings)
        , log_pa(pa)
        , start_(boost::chrono::steady_clock::now())
        , row_count_(0)
        , contains_logger_(contains_logger)
    {
    }

    ~request_profiler()
    {
        if (log_timings || log_pa)
        {
            boost::chrono::milliseconds delay =
                boost::chrono::duration_cast<boost::chrono::milliseconds>(
                    boost::chrono::steady_clock::now() - start_);
            if (log_timings)
            {
                YLOG_CTX(logger(), ctx_, info)
                    << log_db(db_) << " ymod_pq request " << name_ << " finished in " << delay
                    << " with " << row_count_ << " rows";
            }
            if (log_pa)
            {
                pa::async_profiler::add(
                    pa::postgresql,
                    "",
                    name_,
                    ctx_->uniq_id(),
                    static_cast<std::uint32_t>(delay.count()));
            }
        }
    }

    yplatform::log::source& logger()
    {
        return contains_logger_->logger();
    }

    void set_row_count(size_t n)
    {
        row_count_ = n;
    }

private:
    yplatform::task_context_ptr ctx_;
    const std::string& name_;
    const std::string& db_;
    bool log_timings;
    bool log_pa;
    boost::chrono::steady_clock::time_point start_;
    size_t row_count_;
    contains_logger_ptr contains_logger_;
};

struct request_args
{
    yplatform::task_context_ptr ctx;
    ymod_pq::settings_ptr settings;
    std::string db;
    std::string request;
    bind_array_ptr vars;
    yplatform::time_traits::duration deadline;
    boost::shared_ptr<request_profiler> profiler;
    boost::shared_ptr<apq::query> query;
};

typedef boost::shared_ptr<request_args> request_args_ptr_t;

template <class Prom>
void log_error_and_set_promise(
    request_args_ptr_t args,
    Prom prom,
    const char* name,
    const char* reason)
{
    auto& logger = args->profiler->logger();
    YLOG_CTX(logger, args->ctx, error)
        << log_db(args->db) << "ymod_pq error"
        << " can`t call " << name << " request " << args->request << " reason " << reason
        << " query text " << args->query->text_;
    std::ostringstream err;
    err << "can't call " << name << " \"" << args->request << "\"";
    prom.set_exception(std::runtime_error(err.str()));
}

boost::shared_ptr<apq::query> rebind_query(const std::string& text, const bind_array_ptr& vars)
{
    boost::shared_ptr<apq::query> query(new apq::query(text));

    if (vars)
    {
        for (std::size_t i = 0; i < vars->size(); ++i)
        {
            switch (vars->type(i))
            {
            case bind_array::STRING_ARRAY:
                query->bind_cref_string_vector(vars->value<std::vector<std::string>>(i));
                break;

            case bind_array::INT64_ARRAY:
                query->bind_cref_int64_vector(vars->value<std::vector<int64_t>>(i));
                break;

            case bind_array::BYTE_ARRAY:
            {
                boost::shared_ptr<cptr_bind_value> v(vars->value_holder<cptr_bind_value>(i));
                query->bind_cptr_byte_array(v->ptr_, v->sz_);
            }
            break;

            case bind_array::ORA_NULL:
                query->bind_null();
                break;

            case bind_array::DATE:
                // We add UTC offset here so PG won't mistake POSIX time for local.
                query->bind_const_string(vars->value_string(i) + "+0000");
                break;

            default:
                query->bind_const_string(vars->value_string(i));
                break;
            }
        }
    }

    return query;
}

void handle_request(
    request_args_ptr_t args,
    response_handler_ptr handler,
    promise_result prom,
    const apq::result& res,
    apq::row_iterator it)
{
    if (res.code())
    {
        log_error_and_set_promise(args, prom, "request", res.message().c_str());
        return;
    }

    if (it == apq::row_iterator())
    {
        prom.set(true);
        return;
    }

    int row = 0;
    for (; it != apq::row_iterator(); ++it)
    {
        handler->handle_row_begin(row);
        for (int i = 0; i < it->size(); ++i)
            handler->handle_cell(row, i, it->at(i), it->is_null(i));
        handler->handle_row_end(row++);
    }

    args->profiler->set_row_count(row);

    prom.set(true);
}

void handle_update(
    request_args_ptr_t args,
    promise_up_result prom,
    const apq::result& res,
    int count)
{
    if (res.code())
    {
        log_error_and_set_promise(args, prom, "update", res.message().c_str());
        return;
    }

    prom.set(count);
}

void handle_execute(request_args_ptr_t args, promise_result prom, const apq::result& res)
{
    if (res.code())
    {
        log_error_and_set_promise(args, prom, "execute", res.message().c_str());
        return;
    }

    prom.set(true);
}

} // namespace

future_result call_impl::request(
    yplatform::task_context_ptr ctx,
    const std::string& db,
    const std::string& request,
    bind_array_ptr vars,
    response_handler_ptr handler,
    bool log_timings,
    const yplatform::time_traits::duration& deadline)
{
    promise_result prom;

    request_args_ptr_t args(new request_args);
    args->ctx = ctx;
    args->db = db.size() ? db : settings_->conninfo;
    args->settings = settings_;
    args->request = request;
    args->vars = vars;
    args->deadline = deadline;
    args->profiler.reset(new request_profiler(
        ctx, args->request, args->db, log_timings, settings_->log_pa, contains_logger_));
    args->query = rebind_query(settings_->queries.request(request), vars);

    boost::shared_ptr<apq::connection_pool> pool = find_pool(db);
    if (!pool)
    {
        log_error_and_set_promise(args, prom, "request", "pool does not exist");
    }
    else
    {
        pool->async_request(
            ctx,
            *args->query,
            boost::bind(handle_request, args, handler, prom, _1, _2),
            apq::result_format_text,
            deadline != yplatform::time_traits::duration::max() ? deadline :
                                                                  settings_->default_deadline);
    }

    return prom;
}

future_up_result call_impl::update(
    yplatform::task_context_ptr ctx,
    const std::string& db,
    const std::string& request,
    bind_array_ptr vars,
    bool log_timings,
    const yplatform::time_traits::duration& deadline)
{
    promise_up_result prom;

    request_args_ptr_t args(new request_args);
    args->ctx = ctx;
    args->settings = settings_;
    args->db = db.size() ? db : settings_->conninfo;
    args->request = request;
    args->vars = vars;
    args->deadline = deadline;
    args->profiler.reset(new request_profiler(
        ctx, args->request, args->db, log_timings, settings_->log_pa, contains_logger_));
    args->query = rebind_query(settings_->queries.request(request), vars);

    boost::shared_ptr<apq::connection_pool> pool = find_pool(db);
    if (!pool)
    {
        log_error_and_set_promise(args, prom, "update", "pool does not exist");
    }
    else
    {
        pool->async_update(
            ctx,
            *args->query,
            boost::bind(handle_update, args, prom, _1, _2),
            deadline != yplatform::time_traits::duration::max() ? deadline :
                                                                  settings_->default_deadline);
    }

    return prom;
}

future_result call_impl::execute(
    yplatform::task_context_ptr ctx,
    const std::string& db,
    const std::string& request,
    bind_array_ptr vars,
    bool log_timings,
    const yplatform::time_traits::duration& deadline)
{
    promise_result prom;

    request_args_ptr_t args(new request_args);
    args->ctx = ctx;
    args->settings = settings_;
    args->db = db.size() ? db : settings_->conninfo;
    args->request = request;
    args->vars = vars;
    args->deadline = deadline;
    args->profiler.reset(new request_profiler(
        ctx, args->request, args->db, log_timings, settings_->log_pa, contains_logger_));
    args->query = rebind_query(settings_->queries.request(request), vars);

    boost::shared_ptr<apq::connection_pool> pool = find_pool(db);
    if (!pool)
    {
        log_error_and_set_promise(args, prom, "execute", "pool does not exist");
    }
    else
    {
        pool->async_execute(
            ctx,
            *args->query,
            boost::bind(handle_execute, args, prom, _1),
            deadline != yplatform::time_traits::duration::max() ? deadline :
                                                                  settings_->default_deadline);
    }

    return prom;
}

std::string msec_str(unsigned long usec)
{
    std::ostringstream os;
    os << (usec / 1000) << '.' << std::setw(3) << std::setfill('0') << (usec % 1000);
    return os.str();
}

yplatform::ptree call_impl::get_stats() const
{
    yplatform::ptree result;

    boost::mutex::scoped_lock lock(mutex_);

    std::size_t pool_index = 0;
    for (const auto& pool_pair : pools_)
    {
        const std::string& pool_conninfo = pool_pair.first;
        const pool_ptr& pool = pool_pair.second;

        auto stats = pool->get_stats();

        yplatform::ptree pool_node;

        pool_node.put("conninfo", pool_conninfo);

        pool_node.put("busy", stats.busy_connections);
        pool_node.put("idle", stats.free_connections);
        pool_node.put("pending", stats.pending_connections);
        pool_node.put("max", stats.max_connections);
        pool_node.put("queue_size", stats.queue_size);

        pool_node.put(
            "average_request_roundtrip_msec", msec_str(stats.average_request_roundtrip_usec));
        pool_node.put(
            "average_request_db_latency_msec", msec_str(stats.average_request_db_latency_usec));
        pool_node.put("average_wait_time_msec", msec_str(stats.average_wait_time_usec));

        pool_node.put("dropped_connections.timed_out", stats.num_dropped_connections.timed_out);
        pool_node.put("dropped_connections.failed", stats.num_dropped_connections.failed);
        pool_node.put("dropped_connections.busy", stats.num_dropped_connections.busy);
        pool_node.put("dropped_connections.with_result", stats.num_dropped_connections.with_result);

        std::string pool_name = "pool_" + std::to_string(pool_index++);
        result.put_child(std::move(pool_name), pool_node);
    }

    return result;
}

call_impl::call_impl(yplatform::reactor& reactor, const settings& settings)
    : settings_(new struct settings(settings))
    , reactor_(reactor)
    , contains_logger_(new yplatform::log::contains_logger)
    , closed_(true)
{
    YLOG_L(debug) << "pq_module task instaniated";

    for (size_t i = 0; i < reactor_.size(); ++i)
    {
        auto io = reactor_[i]->io();
        auto& service = boost::asio::use_service<yplatform::net::dns::resolver_service>(*io);
        service.setup_service(settings_->dns);
    }

    // Initialize the default pool.
    pool_.reset(new apq::connection_pool(*reactor_.io(), settings_->stat_window_size));
    pool_->set_conninfo(settings_->conninfo);
    pool_->set_limit(settings_->max_conn);
    pool_->set_async_resolve(settings_->async_resolve);
    pool_->set_ipv6_only(settings_->ipv6_only);
    pool_->set_connect_timeout(settings_->connect_timeout);
    pool_->set_queue_timeout(settings_->queue_timeout);
    pool_->set_idle_timeout(settings_->idle_timeout);

    // Default pool is also available by its conninfo.
    if (!settings_->conninfo.empty()) pools_.insert(std::make_pair(settings_->conninfo, pool_));

    closed_ = false;
}

void call_impl::logger(const yplatform::log::source& source)
{
    contains_logger::logger(source);
    contains_logger_->logger(logger());
}

call_impl::call_impl(yplatform::reactor& reactor, const yplatform::ptree& xml)
    : call_impl(reactor, settings(xml))
{
}

void call_impl::clear()
{
    if (pool_) pool_->clear();
    pool_.reset();

    boost::mutex::scoped_lock lock(mutex_);
    closed_ = true;
    pools_.clear();
}

void call_impl::stop()
{
    clear();
}

call_impl::pool_ptr call_impl::find_pool(const std::string& db)
{
    // If conninfo is not specified we use the default pool.
    if (db.empty()) return pool_;

    // Look up the pool by conninfo.
    boost::mutex::scoped_lock lock(mutex_);
    if (closed_) return call_impl::pool_ptr();
    boost::unordered_map<std::string, pool_ptr>::const_iterator it = pools_.find(db);
    if (it == pools_.end())
    {
        // Create a new pool with the given conninfo. For now conninfo strings
        // for the same pool should be identical.
        pool_ptr pool(new apq::connection_pool(*reactor_.io(), settings_->stat_window_size));
        pool->set_conninfo(db);
        pool->set_limit(settings_->max_conn);
        pool->set_async_resolve(settings_->async_resolve);
        pool->set_ipv6_only(settings_->ipv6_only);
        pool->set_connect_timeout(settings_->connect_timeout);
        pool->set_queue_timeout(settings_->queue_timeout);
        pool->set_idle_timeout(settings_->idle_timeout);
        it = pools_.insert(std::make_pair(db, pool)).first;
    }
    return it->second;
}

} // namespace ymod_pq

#include <yplatform/module_registration.h>
DEFINE_SERVICE_OBJECT(ymod_pq::call_impl)
