#include <fstream>
#include <boost/lexical_cast.hpp>
#include "query_conf.h"
#include "query_conf_parser.h"

//#define TEST_QUERY_CONF_PARSE 1
namespace ymod_pq {

const std::vector<int> db_request::default_;

bool is_comment(const std::string& line)
{
    std::string::size_type i = 0;
    for (; i < line.size() && std::isspace(line[i]); ++i)
        ;
    return (i < line.size() && line[i] == '#');
}

void load_query_conf_file(const std::string& file_name, std::string& buffer)
{
    std::ifstream in(file_name.c_str(), std::ios_base::in);
    if (!in) throw std::runtime_error("query_conf file not found");
    in.unsetf(std::ios::skipws); // No white space skipping!
    std::string local_buff;
    while (std::getline(in, local_buff))
    {
        if (is_comment(local_buff)) continue;
        buffer += local_buff + '\n';
    }
}

void query_conf::parse(const std::string& file_name)
{
    std::string storage; // We will read the contents here.
    load_query_conf_file(file_name, storage);
    query_conf_grammar<std::string::const_iterator> xml; // Our grammar
    raw_query_conf ast;                                  // Our tree

    using boost::spirit::standard_wide::space;
    std::string::const_iterator iter = storage.begin();
    std::string::const_iterator end = storage.end();
    bool r = phrase_parse(iter, end, xml, space, ast);
    if (!r || iter != end)
    {
        throw std::runtime_error("query_conf file syntax error");
    }

    for (std::vector<raw_db_request>::iterator i_req = ast.requests.begin(),
                                               i_req_end = ast.requests.end();
         i_req != i_req_end;
         ++i_req)
    {
        db_request request;
        request.debug_ = (i_req->debug == "1");
        request.results_.swap(i_req->structs);
        for (std::vector<std::string>::const_iterator i_var = i_req->vars.begin(),
                                                      i_var_end = i_req->vars.end();
             i_var != i_var_end;
             ++i_var)
        {
            request.named_args_.insert(std::make_pair(*i_var, std::vector<int>()));
        }
        unsigned bind_num = 1;
        for (std::string::const_iterator i_query = i_req->query.begin(),
                                         i_query_end = i_req->query.end();
             i_query != i_query_end;
             ++i_query)
        {
            if (*i_query != '%')
            {
                request.request_ += *i_query;
            }
            else
            {
                ++i_query;
                if (i_query == i_query_end) break;
                if (*i_query == '%') continue;
                if (*i_query == 'v')
                {
                    ++bind_num;
                    ++i_query;
                    std::string var_name;
                    while (i_query != i_query_end)
                    {
                        var_name += *i_query;
                        if (*i_query == '%') break;
                    }
                    if (i_query == i_query_end)
                    {
                        throw std::runtime_error("query_conf file syntax error");
                    }
                    std::map<std::string, std::vector<int>>::iterator i_var =
                        request.named_args_.find(var_name);
                    if (i_var == request.named_args_.end())
                    {
                        throw std::runtime_error("query_conf file syntax error");
                    }
                    i_var->second.push_back(bind_num);
                    request.request_ += "$" + boost::lexical_cast<std::string>(bind_num);
                }
                else
                {
                    request.request_ += "$" + boost::lexical_cast<std::string>(bind_num++);
                }
            }
        }
        requests_.insert(std::make_pair(i_req->name, request));
#ifdef TEST_QUERY_CONF_PARSE
        std::cout << "*******************************************\n";
        std::cout << i_req->name << ": " << request.request_ << "\n";
        std::cout << "*******************************************\n";
#endif
    }
}

}

#ifdef TEST_QUERY_CONF_PARSE

int main(int argc, char** argv)
{
    char const* filename;
    if (argc > 1)
    {
        filename = argv[1];
    }
    else
    {
        std::cerr << "Error: No input file provided." << std::endl;
        return 1;
    }
    ymod_pq::query_conf conf;
    conf.parse(filename);
    return 0;
}

#endif
